/* JAI-Ext - OpenSource Java Advanced Image Extensions Library
 *    http://www.geo-solutions.it/
 *    Copyright 2018 GeoSolutions
 *    
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


/*
 * Copyright (c) 2018, Michael Bedward. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package it.geosolutions.jaiext.jiffle.parser.node;

import java.util.Objects;

import it.geosolutions.jaiext.jiffle.Jiffle;
import it.geosolutions.jaiext.jiffle.Jiffle.RuntimeModel;
import it.geosolutions.jaiext.jiffle.parser.*;


/**
 *
 * @author michael
 */
public class SetDestValue extends Expression {
    private final String destVar;
    private final Expression expr;
    private final Band band;
    
    private static JiffleType ensureScalar(Expression e) throws NodeException {
        JiffleType type = e.getType();
        if (type == JiffleType.D) {
            return type;
        }
        throw new NodeException(Errors.EXPECTED_SCALAR);
    }

    public SetDestValue(String varName, Expression expr)
            throws NodeException {
        this(varName, new Band(new IntLiteral("0")), expr);
    }

    public SetDestValue(String varName, Band band, Expression expr) 
            throws NodeException {
        
        super(ensureScalar(expr));
        
        this.destVar = varName;
        this.expr = expr;
        this.band = band;
    }
    
    @Override
    public String toString() {
        return DirectSources.setDestValue(RuntimeModel.DIRECT, destVar, expr.toString());
    }

    public void write(SourceWriter w) {
        RuntimeModel runtimeModel = w.getRuntimeModel();
        switch (runtimeModel) {
            case DIRECT:
                if (w.isInternalBaseClass()) {
                    w.append("d_").append(destVar).append(".write(_x, _y, ");
                    if (band != null) {
                      w.append(band);
                    } else {
                      w.append("0");  
                    }
                    w.append(", ").append(expr).append(")");                    
                } else {
                    w.append("writeToImage(\"").append(destVar).append("\", _x, _y, ");
                    if (band != null) {
                        w.append(band);
                    } else {
                        w.append("0");
                    }
                    w.append(", ").append(expr).append(")");
                }
                
                break;
                
            case INDIRECT:
                w.append("result[");
                if (band != null) {
                    w.append(band);
                } else {
                    w.append("0");
                }
                w.append("] = ").append(expr);
                break;

            default:
                throw new IllegalArgumentException("Invalid runtime model: " + runtimeModel);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        SetDestValue that = (SetDestValue) o;
        return Objects.equals(destVar, that.destVar) &&
                Objects.equals(expr, that.expr);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), destVar, expr);
    }

    public String getDestVar() {
        return destVar;
    }

    public Expression getExpr() {
        return expr;
    }
}
