/*
 * Decompiled with CFR 0.152.
 */
package org.geoserver.geofence.util;

import java.util.function.BiFunction;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.geoserver.catalog.CatalogInfo;
import org.geoserver.catalog.LayerGroupInfo;
import org.geoserver.catalog.LayerInfo;
import org.geoserver.catalog.ResourceInfo;
import org.geotools.api.referencing.FactoryException;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;
import org.geotools.api.referencing.operation.MathTransform;
import org.geotools.api.referencing.operation.TransformException;
import org.geotools.geometry.jts.JTS;
import org.geotools.referencing.CRS;
import org.geotools.util.logging.Logging;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.io.ParseException;
import org.locationtech.jts.io.WKTReader;

public class GeomHelper {
    private static final Logger LOGGER = Logging.getLogger(GeomHelper.class);

    public static Geometry parseWKT(String wkt) {
        Geometry result = null;
        if (wkt != null) {
            WKTReader wktReader = new WKTReader();
            try {
                if (wkt.contains("SRID")) {
                    String[] areaParts = wkt.split(";");
                    result = wktReader.read(areaParts[1]);
                    int srid = Integer.valueOf(areaParts[0].split("=")[1]);
                    result.setSRID(srid);
                } else {
                    result = wktReader.read(wkt);
                    result.setSRID(4326);
                }
            }
            catch (ParseException e) {
                throw new RuntimeException("Failed to unmarshal the area wkt", e);
            }
        }
        return result;
    }

    public static Geometry reprojectAndUnion(Geometry first, Geometry second, boolean lessRestrictive) {
        BiFunction<Geometry, Geometry, Geometry> union = (g1, g2) -> g1.union(g2);
        if (lessRestrictive) {
            return GeomHelper.reprojectAndApplyOpFavourNull(first, second, union);
        }
        return GeomHelper.reprojectAndApplyOperation(first, second, union);
    }

    public static Geometry reprojectAndIntersect(Geometry first, Geometry second, boolean favourNull) {
        BiFunction<Geometry, Geometry, Geometry> intersection = (g1, g2) -> g1.intersection(g2);
        if (favourNull) {
            return GeomHelper.reprojectAndApplyOpFavourNull(first, second, intersection);
        }
        return GeomHelper.reprojectAndApplyOperation(first, second, intersection);
    }

    public static Geometry reprojectAndIntersect(Geometry g1, Geometry g2) {
        return GeomHelper.reprojectAndIntersect(g1, g2, false);
    }

    public static Geometry reprojectAndApplyOpFavourNull(Geometry first, Geometry second, BiFunction<Geometry, Geometry, Geometry> operation) {
        if (first == null || second == null) {
            return null;
        }
        return GeomHelper.reprojectAndApplyOperation(first, second, operation);
    }

    public static Geometry reprojectAndApplyOperation(Geometry first, Geometry second, BiFunction<Geometry, Geometry, Geometry> operation) {
        if (first == null) {
            return second;
        }
        if (second == null) {
            return first;
        }
        if (first.getSRID() != second.getSRID()) {
            try {
                CoordinateReferenceSystem target = CRS.decode((String)("EPSG:" + first.getSRID()), (boolean)true);
                CoordinateReferenceSystem source = CRS.decode((String)("EPSG:" + second.getSRID()), (boolean)true);
                MathTransform transformation = CRS.findMathTransform((CoordinateReferenceSystem)source, (CoordinateReferenceSystem)target);
                second = JTS.transform((Geometry)second, (MathTransform)transformation);
                second.setSRID(first.getSRID());
            }
            catch (FactoryException | TransformException e) {
                throw new RuntimeException("Unable to intersect allowed areas: error during transformation from " + second.getSRID() + " to " + first.getSRID());
            }
        }
        Geometry result = operation.apply(first, second);
        result.setSRID(first.getSRID());
        return result;
    }

    public static Geometry reprojectGeometry(Geometry geometry, CoordinateReferenceSystem targetCRS) {
        if (geometry == null) {
            return null;
        }
        try {
            CoordinateReferenceSystem geomCrs = CRS.decode((String)("EPSG:" + geometry.getSRID()), (boolean)true);
            if (targetCRS != null && !CRS.equalsIgnoreMetadata((Object)geomCrs, (Object)targetCRS)) {
                MathTransform mt = CRS.findMathTransform((CoordinateReferenceSystem)geomCrs, (CoordinateReferenceSystem)targetCRS, (boolean)true);
                geometry = JTS.transform((Geometry)geometry, (MathTransform)mt);
                Integer srid = CRS.lookupEpsgCode((CoordinateReferenceSystem)targetCRS, (boolean)false);
                geometry.setSRID(srid.intValue());
            }
            return geometry;
        }
        catch (FactoryException | TransformException e) {
            LOGGER.log(Level.SEVERE, "Error while reprojecting geometry: " + e.getMessage(), e);
            throw new RuntimeException(e);
        }
    }

    public static CoordinateReferenceSystem getCRSFromInfo(CatalogInfo info) {
        CoordinateReferenceSystem crs = null;
        if (info instanceof LayerInfo) {
            crs = ((LayerInfo)info).getResource().getCRS();
        } else if (info instanceof ResourceInfo) {
            crs = ((ResourceInfo)info).getCRS();
        } else if (info instanceof LayerGroupInfo) {
            crs = ((LayerGroupInfo)info).getBounds().getCoordinateReferenceSystem();
        } else {
            throw new RuntimeException("Cannot retrieve CRS from info " + info.getClass().getSimpleName());
        }
        return crs;
    }
}

