/*
 * Decompiled with CFR 0.152.
 */
package org.geoserver.platform.resource;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.geoserver.platform.resource.ResourceListener;
import org.geoserver.platform.resource.ResourceNotification;
import org.geoserver.platform.resource.ResourceNotificationDispatcher;
import org.geotools.util.logging.Logging;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.scheduling.concurrent.CustomizableThreadFactory;

public class FileSystemWatcher
implements ResourceNotificationDispatcher,
DisposableBean {
    private static final Logger LOGGER = Logging.getLogger(FileSystemWatcher.class);
    private ScheduledExecutorService pool;
    private final Function<String, File> fileExtractor;
    protected long lastmodified;
    CopyOnWriteArrayList<Watch> watchers = new CopyOnWriteArrayList();
    private Runnable sync = new Runnable(){

        @Override
        public void run() {
            long now = System.currentTimeMillis();
            for (Watch watch : FileSystemWatcher.this.watchers) {
                Delta delta;
                if (watch.getListeners().isEmpty()) {
                    FileSystemWatcher.this.watchers.remove(watch);
                    continue;
                }
                boolean directory = watch.file.isDirectory();
                Level level = Level.FINER;
                long start = System.nanoTime();
                if (directory) {
                    LOGGER.log(level, "polling contents of " + watch.file);
                }
                try {
                    delta = watch.changed(now);
                }
                catch (RuntimeException e) {
                    LOGGER.log(Level.WARNING, "Error polling contents of " + watch.file, e);
                    return;
                }
                if (directory && LOGGER.isLoggable(level)) {
                    long ellapsedMicros = TimeUnit.MICROSECONDS.convert(System.nanoTime() - start, TimeUnit.NANOSECONDS);
                    long ellapsedMillis = TimeUnit.MILLISECONDS.convert(ellapsedMicros, TimeUnit.MICROSECONDS);
                    String unit = ellapsedMillis == 0L ? "us" : "ms";
                    long time = ellapsedMillis == 0L ? ellapsedMicros : ellapsedMillis;
                    LOGGER.log(level, String.format("delta computed in %,d%s for %s", time, unit, watch.file));
                }
                if (delta == null) continue;
                this.notify(watch, delta);
            }
        }

        private void notify(Watch watch, Delta delta) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.info(String.format("Notifying %s change on %s. Created: %,d, removed: %,d, modified: %,d", new Object[]{delta.kind, delta.context, delta.created.size(), delta.removed.size(), delta.modified.size()}));
            }
            CompletableFuture.runAsync(() -> {
                List<ResourceNotification.Event> events = ResourceNotification.delta(watch.file, delta.created, delta.removed, delta.modified);
                ResourceNotification notify = new ResourceNotification(watch.getPath(), delta.kind, watch.last, events);
                for (ResourceListener listener : watch.getListeners()) {
                    try {
                        listener.changed(notify);
                    }
                    catch (Throwable t) {
                        Logger logger = Logger.getLogger(listener.getClass().getPackage().getName());
                        logger.log(Level.FINE, "Unable to notify " + watch + ":" + t.getMessage(), t);
                    }
                }
            });
        }
    };
    private ScheduledFuture<?> monitor;
    private TimeUnit unit = TimeUnit.SECONDS;
    private long delay = 5L;
    private static CustomizableThreadFactory tFactory = new CustomizableThreadFactory("FileSystemWatcher-");

    FileSystemWatcher(Function<String, File> fileExtractor) {
        Objects.requireNonNull(fileExtractor);
        this.pool = Executors.newSingleThreadScheduledExecutor((ThreadFactory)tFactory);
        this.fileExtractor = fileExtractor;
    }

    FileSystemWatcher() {
        this(path -> new File(path.replace('/', File.separatorChar)));
    }

    private Watch watch(File file, String path) {
        Objects.requireNonNull(file);
        Objects.requireNonNull(path);
        for (Watch watch : this.watchers) {
            if (!watch.isMatch(file, path)) continue;
            return watch;
        }
        return null;
    }

    @Override
    public synchronized void addListener(String path, ResourceListener listener) {
        Objects.requireNonNull(path, "Path for notification is required");
        File file = this.fileExtractor.apply(path);
        Objects.requireNonNull(file, "File to watch is required");
        Watch watch = this.watch(file, path);
        if (watch == null) {
            watch = new Watch(file, path);
            this.watchers.add(watch);
            if (this.monitor == null) {
                this.monitor = this.pool.scheduleWithFixedDelay(this.sync, this.delay, this.delay, this.unit);
            }
        }
        watch.addListener(listener);
    }

    @Override
    public synchronized boolean removeListener(String path, ResourceListener listener) {
        Objects.requireNonNull(path, "Path for notification is required");
        File file = this.fileExtractor.apply(path);
        Objects.requireNonNull(file, "File to watch is required");
        Watch watch = this.watch(file, path);
        boolean removed = false;
        if (watch != null) {
            watch.removeListener(listener);
            if (watch.getListeners().isEmpty()) {
                removed = this.watchers.remove(watch);
            }
        }
        if (removed && this.watchers.isEmpty() && this.monitor != null) {
            this.monitor.cancel(false);
            this.monitor = null;
        }
        return removed;
    }

    public void schedule(long delay, TimeUnit unit) {
        this.delay = delay;
        this.unit = unit;
        if (this.monitor != null) {
            this.monitor.cancel(false);
            this.monitor = this.pool.scheduleWithFixedDelay(this.sync, delay, delay, unit);
        }
    }

    public void destroy() throws Exception {
        this.pool.shutdown();
        this.monitor = null;
    }

    @Override
    public void changed(ResourceNotification notification) {
        throw new UnsupportedOperationException();
    }

    static {
        tFactory.setDaemon(true);
    }

    private class Watch
    implements Comparable<Watch> {
        final File file;
        final String path;
        final List<ResourceListener> listeners = new CopyOnWriteArrayList<ResourceListener>();
        long last = 0L;
        boolean exsists;
        private Set<File> children = null;
        private long childrenLastModifiedMax = 0L;

        public Watch(File file, String path) {
            Objects.requireNonNull(file);
            Objects.requireNonNull(path);
            this.file = file;
            this.path = path;
            this.exsists = file.exists();
            long l = this.last = this.exsists ? file.lastModified() : 0L;
            if (file.isDirectory()) {
                this.children = this.loadDirectoryContents(file);
                this.childrenLastModifiedMax = this.children.parallelStream().mapToLong(File::lastModified).max().orElse(0L);
            }
        }

        private Set<File> loadDirectoryContents(File directory) {
            File[] files = directory.listFiles();
            if (files == null) {
                return new HashSet<File>();
            }
            return Arrays.stream(files).collect(Collectors.toSet());
        }

        public void addListener(ResourceListener listener) {
            this.listeners.add(listener);
        }

        public void removeListener(ResourceListener listener) {
            this.listeners.remove(listener);
        }

        public String getPath() {
            return this.path;
        }

        public List<ResourceListener> getListeners() {
            return this.listeners;
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + this.file.hashCode();
            result = 31 * result + this.path.hashCode();
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof Watch)) {
                return false;
            }
            Watch other = (Watch)obj;
            return this.file.equals(other.file) && this.path.equals(other.path);
        }

        public String toString() {
            return "Watch [path=" + this.path + ", file=" + this.file + ", listeners=" + this.listeners.size() + "]";
        }

        @Override
        public int compareTo(Watch other) {
            return this.path.compareTo(other.path);
        }

        public Delta changed(long now) {
            if (!this.file.exists()) {
                return this.watchedFileRemoved(now);
            }
            if (this.file.isFile()) {
                return this.simpleFileCheck();
            }
            return this.pollDirectory();
        }

        private Delta pollDirectory() {
            ResourceNotification.Kind kind;
            long fileModified = this.file.lastModified();
            if (this.exsists) {
                kind = ResourceNotification.Kind.ENTRY_MODIFY;
            } else {
                this.children = new HashSet<File>();
                kind = ResourceNotification.Kind.ENTRY_CREATE;
                this.exsists = true;
            }
            this.last = fileModified;
            long childrenMaxLastModified = this.childrenLastModifiedMax;
            CompletableFuture<File[]> contentsFuture = CompletableFuture.supplyAsync(() -> this.file.listFiles());
            EnumMap<ResourceNotification.Kind, List> itemsByType = new EnumMap<ResourceNotification.Kind, List>(ResourceNotification.Kind.class);
            Iterator<File> it = this.children.iterator();
            while (it.hasNext()) {
                File child = it.next();
                long lastModified = child.lastModified();
                if (0L == lastModified) {
                    it.remove();
                    itemsByType.computeIfAbsent(ResourceNotification.Kind.ENTRY_DELETE, k -> new ArrayList()).add(child.getName());
                    continue;
                }
                if (lastModified <= this.childrenLastModifiedMax) continue;
                childrenMaxLastModified = lastModified;
                itemsByType.computeIfAbsent(ResourceNotification.Kind.ENTRY_MODIFY, k -> new ArrayList()).add(child.getName());
            }
            File[] contents = contentsFuture.join();
            if (null != contents) {
                for (File child : contents) {
                    if (!this.children.add(child)) continue;
                    childrenMaxLastModified = Math.max(childrenMaxLastModified, child.lastModified());
                    itemsByType.computeIfAbsent(ResourceNotification.Kind.ENTRY_CREATE, k -> new ArrayList()).add(child.getName());
                }
            }
            if (itemsByType.isEmpty()) {
                return null;
            }
            this.childrenLastModifiedMax = childrenMaxLastModified;
            List created = (List)itemsByType.get((Object)ResourceNotification.Kind.ENTRY_CREATE);
            List removed = (List)itemsByType.get((Object)ResourceNotification.Kind.ENTRY_DELETE);
            List modified = (List)itemsByType.get((Object)ResourceNotification.Kind.ENTRY_MODIFY);
            Delta delta = new Delta(this.file, kind, created, removed, modified);
            return delta;
        }

        private Delta simpleFileCheck() {
            long fileModified = this.file.lastModified();
            if (fileModified > this.last || !this.exsists) {
                ResourceNotification.Kind kind = this.exsists ? ResourceNotification.Kind.ENTRY_MODIFY : ResourceNotification.Kind.ENTRY_CREATE;
                this.exsists = true;
                this.last = fileModified;
                return new Delta(this.file, kind);
            }
            return null;
        }

        private Delta watchedFileRemoved(long now) {
            Delta delta = null;
            if (this.exsists) {
                if (this.children == null) {
                    delta = new Delta(this.file, ResourceNotification.Kind.ENTRY_DELETE);
                } else {
                    List<String> deleted = this.children.stream().map(File::getName).collect(Collectors.toList());
                    delta = new Delta(this.file, ResourceNotification.Kind.ENTRY_DELETE, null, deleted, null);
                }
                this.last = now;
                this.exsists = false;
                this.children = null;
            }
            return delta;
        }

        public boolean isMatch(File file, String path) {
            return this.file.equals(file) && this.path.equals(path);
        }
    }

    static class Delta {
        final File context;
        final ResourceNotification.Kind kind;
        final List<String> created;
        final List<String> removed;
        final List<String> modified;

        public Delta(File context, ResourceNotification.Kind kind) {
            this.context = context;
            this.kind = kind;
            this.modified = Collections.emptyList();
            this.removed = this.modified;
            this.created = this.modified;
        }

        public Delta(File context, ResourceNotification.Kind kind, List<String> created, List<String> removed, List<String> modified) {
            this.context = context;
            this.kind = kind;
            this.created = created == null ? Collections.emptyList() : created;
            this.removed = removed == null ? Collections.emptyList() : removed;
            this.modified = modified == null ? Collections.emptyList() : modified;
        }

        public int size() {
            return this.created.size() + this.removed.size() + this.modified.size();
        }

        public String toString() {
            return "Delta [context=" + this.context + ", created=" + this.created + ", removed=" + this.removed + ", modified=" + this.modified + "]";
        }
    }
}

