/* (c) 2014 Open Source Geospatial Foundation - all rights reserved
 * (c) 2001 - 2013 OpenPlans
 * This code is licensed under the GPL 2.0 license, available at the root
 * application directory.
 */
package org.geoserver.wms;

import java.io.IOException;
import java.util.Set;
import org.geoserver.ows.Response;
import org.geoserver.platform.ServiceException;

/**
 * Produces and maps for an specific output format.
 *
 * <p>A GetMapOutputFormat implementation is meant to produce one and only one type of content,
 * whose normative MIME-Type is advertised through the {@link #getContentType()} method. Yet, the
 * {@link #getOutputFormatNames()} method is meant to advertise the map format in the capabilities
 * document and may or may not match the MIME-Type.
 *
 * <p>To incorporate a new producer specialized in a given output format, there must be a {@code
 * GetMapOutputFormat} registered in the Spring context that can provide instances of that concrete
 * implementation, as well as a {@link Response} Spring bean that can encode the produced {@link
 * WebMap}. Hence, it's counterpart {@code Response} implementation {@link
 * Response#canHandle(org.geoserver.platform.Operation) canHandle(Operation)} method must be
 * implemented in a consistent way with the output format's {@link #getMimeType()} and {@link
 * #getOutputFormatNames()}.
 *
 * <p>The methods defined in this interface respect the general parse request/produce response/get
 * mime type/write content workflow, so they should raise an exception if are called in the wrong
 * order (which is produceMap -> getContentType -> writeTo)
 *
 * @author Gabriel Roldan
 * @author Simone Giannecchini, GeoSolutions
 * @version $Id$
 */
public interface GetMapOutputFormat {
    /**
     * Asks this map producer to create a map image for the passed {@linkPlain WMSMapContext}, which
     * contains enough information for doing such a process.
     *
     * @throws ServiceException something goes wrong
     */
    public WebMap produceMap(WMSMapContent mapContent) throws ServiceException, IOException;

    /**
     * Returns the list of content type aliases for this output format, that are the ones to be used
     * as Format elements in the GetCapabilities document.
     *
     * <p>Aliases are used to easy the task of writing a GetMap request, (for example, to type
     * &outputformat=svg instead of the full &outputformat=image/svg+xml)
     *
     * @return the aliases a map of the content type this map producer creates content type can be
     *     asked by through a GetMap request.
     */
    public Set<String> getOutputFormatNames();

    /** Returns the mime type generated by the format */
    public String getMimeType();

    /** Returns the capabilities for the specified format */
    public MapProducerCapabilities getCapabilities(String format);
}
