/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 *
 *    (C) 2008-2011, Open Source Geospatial Foundation (OSGeo)
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 */

package org.geotools.data.complex.config;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.geotools.util.CheckedArrayList;
import org.geotools.util.CheckedHashMap;
import org.geotools.util.CheckedHashSet;

/**
 * Configuration object for a {@link org.geotools.data.complex.AppSchemaDataAccess}.
 *
 * <p>This configuration object contains all the needed elements for a AppSchemaDataAccess to aquire the source and
 * target FeatureTypes, and apply the mappings between attributes to serve community schemas.
 *
 * @author Gabriel Roldan (Axios Engineering)
 * @version $Id$
 * @since 2.4
 */
public class AppSchemaDataAccessDTO implements Serializable {
    private static final long serialVersionUID = -8649884546130644668L;

    /** Url used as base path to resolve relative file names in {@link #targetSchemasUris} */
    private String baseSchemasUrl;

    /** Mapping of prefix/namespace used in the target schema */
    private Map<String, String> namespaces = Collections.emptyMap();

    /**
     * List of configuration objects used to aquire the datastores that provides the source FeatureTypes. Source feature
     * types are those internally defined whose Feature instances are converted to features of the target schemas by
     * applying the FeatureTypeMappings.
     */
    private List<SourceDataStore> sourceDataStores = Collections.emptyList();

    private Set<TypeMapping> typeMappings = Collections.emptySet();

    /** List of file names, that may be fully qualified URL's, or paths relative to {@link #baseSchemasUrl} */
    private List<String> targetSchemasUris = Collections.emptyList();

    /**
     * List of the paths of other related types that are mapped separately that shouldn't be visible on their own, thus
     * included in "include" statement
     */
    private List<String> includes = Collections.emptyList();

    private String oasisCatalogUri;

    /** */
    public List<String> getTargetSchemasUris() {
        return new ArrayList<>(targetSchemasUris);
    }

    /** */
    public void setTargetSchemasUris(List<String> targetSchemasUris) {
        this.targetSchemasUris = new CheckedArrayList<>(String.class);

        if (targetSchemasUris != null) {
            this.targetSchemasUris.addAll(targetSchemasUris);
        }
    }

    /** */
    public void setNamespaces(Map<String, String> nameSpaces) {
        if (nameSpaces == null) {
            this.namespaces = Collections.emptyMap();
        } else {
            this.namespaces = new CheckedHashMap<>(String.class, String.class);
            this.namespaces.putAll(nameSpaces);
        }
    }

    /** */
    public Map<String, String> getNamespaces() {
        return new HashMap<>(namespaces);
    }

    /** Set the path of other related types that are mapped separately */
    public void setIncludedTypes(ArrayList<String> includes) {
        if (includes != null) {
            this.includes = new CheckedArrayList<>(String.class);
            this.includes.addAll(includes);
        }
    }

    /** Return the list of paths of related types that are mapped separately */
    public List<String> getIncludes() {
        return includes;
    }

    /** */
    public void setSourceDataStores(List<SourceDataStore> dataStores) {
        if (dataStores == null) {
            this.sourceDataStores = Collections.emptyList();
        } else {
            this.sourceDataStores = new CheckedArrayList<>(SourceDataStore.class);
            this.sourceDataStores.addAll(dataStores);
        }
    }

    /** */
    public List<SourceDataStore> getSourceDataStores() {
        return new ArrayList<>(sourceDataStores);
    }

    /** */
    public void setTypeMappings(Set<TypeMapping> typeMappings) {
        this.typeMappings = new CheckedHashSet<>(TypeMapping.class);

        if (typeMappings != null) {
            this.typeMappings.addAll(typeMappings);
        }
    }

    /** */
    public Set<TypeMapping> getTypeMappings() {
        return new HashSet<>(typeMappings);
    }

    /**
     * @return a non null URL for the base location of the resource files in order to serve as the base to resolve
     *     relative configuration paths.
     */
    public String getBaseSchemasUrl() {
        return baseSchemasUrl;
    }

    /**
     * @param baseSchemasUrl URL of a resource which's going to be taken as the base location to resolve configuration
     *     path elements expressed as relative paths.
     */
    public void setBaseSchemasUrl(final String baseSchemasUrl) {
        this.baseSchemasUrl = baseSchemasUrl;
    }

    public String getCatalog() {
        return oasisCatalogUri;
    }

    public void setCatalog(final String oasisCatalogUri) {
        this.oasisCatalogUri = oasisCatalogUri;
    }
}
