/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.data.elasticsearch;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import org.elasticsearch.common.joda.Joda;
import org.geotools.data.FeatureReader;
import org.geotools.data.elasticsearch.ElasticAggregation;
import org.geotools.data.elasticsearch.ElasticDataStore;
import org.geotools.data.elasticsearch.ElasticDataStoreFactory;
import org.geotools.data.elasticsearch.ElasticHit;
import org.geotools.data.elasticsearch.ElasticParserUtil;
import org.geotools.data.elasticsearch.ElasticResponse;
import org.geotools.data.store.ContentState;
import org.geotools.feature.simple.SimpleFeatureBuilder;
import org.geotools.util.logging.Logging;
import org.joda.time.format.DateTimeFormatter;
import org.locationtech.jts.geom.Geometry;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.feature.type.AttributeDescriptor;

class ElasticFeatureReader
implements FeatureReader<SimpleFeatureType, SimpleFeature> {
    private static final Logger LOGGER = Logging.getLogger(ElasticFeatureReader.class);
    private final ContentState state;
    private final SimpleFeatureType featureType;
    private final float maxScore;
    private final ObjectMapper mapper;
    private final ElasticDataStore.ArrayEncoding arrayEncoding;
    private SimpleFeatureBuilder builder;
    private Iterator<ElasticHit> searchHitIterator;
    private Iterator<Map<String, Object>> aggregationIterator;
    private final ElasticParserUtil parserUtil;

    public ElasticFeatureReader(ContentState contentState, ElasticResponse response) {
        this(contentState, response.getHits(), response.getAggregations(), response.getMaxScore());
    }

    public ElasticFeatureReader(ContentState contentState, List<ElasticHit> hits, Map<String, ElasticAggregation> aggregations, float maxScore) {
        this.state = contentState;
        this.featureType = this.state.getFeatureType();
        this.searchHitIterator = hits.iterator();
        this.builder = new SimpleFeatureBuilder(this.featureType);
        this.parserUtil = new ElasticParserUtil();
        this.maxScore = maxScore;
        this.aggregationIterator = Collections.emptyIterator();
        if (aggregations != null && !aggregations.isEmpty()) {
            String aggregationName = aggregations.keySet().stream().findFirst().orElse(null);
            if (aggregations.size() > 1) {
                LOGGER.info("Result has multiple aggregations. Using " + aggregationName);
            }
            if (aggregations.get(aggregationName).getBuckets() != null) {
                this.aggregationIterator = aggregations.get(aggregationName).getBuckets().iterator();
            }
        }
        if (contentState.getEntry() != null && contentState.getEntry().getDataStore() != null) {
            ElasticDataStore dataStore = (ElasticDataStore)contentState.getEntry().getDataStore();
            this.arrayEncoding = dataStore.getArrayEncoding();
        } else {
            this.arrayEncoding = ElasticDataStore.ArrayEncoding.valueOf((String)ElasticDataStoreFactory.ARRAY_ENCODING.getDefaultValue());
        }
        this.mapper = new ObjectMapper();
    }

    public SimpleFeatureType getFeatureType() {
        return this.featureType;
    }

    public SimpleFeature next() {
        String id;
        if (this.searchHitIterator.hasNext()) {
            id = this.nextHit();
        } else {
            this.nextAggregation();
            id = null;
        }
        return this.builder.buildFeature(id);
    }

    private String nextHit() {
        Float relativeScore;
        Float score;
        ElasticHit hit = this.searchHitIterator.next();
        SimpleFeatureType type = this.getFeatureType();
        Map<String, Object> source = hit.getSource();
        if (hit.getScore() != null && !Float.isNaN(hit.getScore().floatValue()) && this.maxScore > 0.0f) {
            score = hit.getScore();
            relativeScore = Float.valueOf(score.floatValue() / this.maxScore);
        } else {
            score = null;
            relativeScore = null;
        }
        for (AttributeDescriptor descriptor : type.getAttributeDescriptors()) {
            String name = descriptor.getType().getName().getLocalPart();
            String sourceName = (String)descriptor.getUserData().get("full_name");
            List<Object> values = hit.field(sourceName);
            if (values == null && source != null) {
                values = this.parserUtil.readField(source, sourceName);
            }
            if (values == null && sourceName.equals("_id")) {
                this.builder.set(name, (Object)hit.getId());
                continue;
            }
            if (values == null && sourceName.equals("_index")) {
                this.builder.set(name, (Object)hit.getIndex());
                continue;
            }
            if (values == null && sourceName.equals("_type")) {
                this.builder.set(name, (Object)hit.getType());
                continue;
            }
            if (values == null && sourceName.equals("_score")) {
                this.builder.set(name, (Object)score);
                continue;
            }
            if (values == null && sourceName.equals("_relative_score")) {
                this.builder.set(name, (Object)relativeScore);
                continue;
            }
            if (values != null && Geometry.class.isAssignableFrom(descriptor.getType().getBinding())) {
                if (values.size() == 1) {
                    this.builder.set(name, (Object)this.parserUtil.createGeometry(values.get(0)));
                    continue;
                }
                this.builder.set(name, (Object)this.parserUtil.createGeometry(values));
                continue;
            }
            if (values != null && Date.class.isAssignableFrom(descriptor.getType().getBinding())) {
                Object dataVal = values.get(0);
                if (dataVal instanceof Double) {
                    this.builder.set(name, (Object)new Date(Math.round((Double)dataVal)));
                    continue;
                }
                if (dataVal instanceof Integer) {
                    this.builder.set(name, (Object)new Date(((Integer)dataVal).longValue()));
                    continue;
                }
                if (dataVal instanceof Long) {
                    this.builder.set(name, (Object)new Date((Long)dataVal));
                    continue;
                }
                List validFormats = (List)descriptor.getUserData().get("date_format");
                DateTimeFormatter dateFormatter = null;
                Date date = null;
                if (validFormats != null && !validFormats.isEmpty()) {
                    for (String format : validFormats) {
                        try {
                            dateFormatter = Joda.forPattern(format).parser();
                            date = dateFormatter.parseDateTime((String)dataVal).toDate();
                            break;
                        }
                        catch (Exception e) {
                            LOGGER.fine("Unable to parse date format ('" + format + "') for " + descriptor);
                        }
                    }
                }
                if (dateFormatter == null) {
                    LOGGER.fine("Unable to find any valid date format for " + descriptor);
                    dateFormatter = Joda.forPattern("date_optional_time").parser();
                }
                if (date == null) {
                    LOGGER.fine("Unable to find any valid date format for " + dataVal);
                    date = dateFormatter.parseDateTime((String)dataVal).toDate();
                }
                this.builder.set(name, date);
                continue;
            }
            if (values != null && values.size() == 1) {
                this.builder.set(name, values.get(0));
                continue;
            }
            if (values == null || name.equals("_aggregation")) continue;
            Object value = this.arrayEncoding == ElasticDataStore.ArrayEncoding.CSV ? values.get(0) : values;
            this.builder.set(name, value);
        }
        return this.state.getEntry().getTypeName() + "." + hit.getId();
    }

    private void nextAggregation() {
        Map<String, Object> aggregation = this.aggregationIterator.next();
        try {
            byte[] data = this.mapper.writeValueAsBytes(aggregation);
            this.builder.set("_aggregation", (Object)data);
        }
        catch (IOException e) {
            LOGGER.warning("Unable to set aggregation. Try reloading layer.");
        }
    }

    public boolean hasNext() {
        return this.searchHitIterator.hasNext() || this.aggregationIterator.hasNext();
    }

    public void close() {
        this.builder = null;
        this.searchHitIterator = null;
    }
}

