/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.map;

import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.geom.NoninvertibleTransformException;
import java.awt.geom.Point2D;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.map.MapBoundsEvent;
import org.geotools.map.MapBoundsListener;
import org.geotools.referencing.CRS;
import org.geotools.util.logging.Logging;
import org.locationtech.jts.geom.Envelope;

public class MapViewport {
    protected static final Logger LOGGER = Logging.getLogger(MapViewport.class);
    private final AtomicBoolean editable;
    private Rectangle screenArea;
    private ReferencedEnvelope bounds;
    private AffineTransform screenToWorld;
    private AffineTransform worldToScreen;
    private CopyOnWriteArrayList<MapBoundsListener> boundsListeners;
    private boolean matchingAspectRatio;
    private boolean hasCenteringTransforms;
    private boolean fixedBoundsOnResize = false;
    private final ReadWriteLock lock = new ReentrantReadWriteLock();

    public MapViewport() {
        this(false);
    }

    public MapViewport(boolean matchAspectRatio) {
        this(null, matchAspectRatio);
    }

    public MapViewport(ReferencedEnvelope bounds) {
        this(bounds, false);
    }

    public MapViewport(ReferencedEnvelope bounds, boolean matchAspectRatio) {
        this.editable = new AtomicBoolean(true);
        this.screenArea = new Rectangle();
        this.hasCenteringTransforms = false;
        this.matchingAspectRatio = matchAspectRatio;
        this.copyBounds(bounds);
        this.setTransforms(true);
    }

    public MapViewport(MapViewport sourceViewport) {
        this.editable = new AtomicBoolean(true);
        this.matchingAspectRatio = sourceViewport.matchingAspectRatio;
        this.copyBounds(sourceViewport.bounds);
        this.doSetScreenArea(sourceViewport.screenArea);
        this.setTransforms(true);
    }

    public boolean isEditable() {
        return this.editable.get();
    }

    public void setEditable(boolean editable) {
        this.editable.set(editable);
    }

    public void setMatchingAspectRatio(boolean enabled) {
        this.lock.writeLock().lock();
        try {
            if (this.checkEditable("setMatchingAspectRatio") && enabled != this.matchingAspectRatio) {
                this.matchingAspectRatio = enabled;
                this.setTransforms(true);
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    public boolean isMatchingAspectRatio() {
        this.lock.readLock().lock();
        try {
            boolean bl = this.matchingAspectRatio;
            return bl;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    public void addMapBoundsListener(MapBoundsListener listener) {
        this.lock.writeLock().lock();
        try {
            if (this.boundsListeners == null) {
                this.boundsListeners = new CopyOnWriteArrayList();
            }
            if (!this.boundsListeners.contains(listener)) {
                this.boundsListeners.add(listener);
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    public void removeMapBoundsListener(MapBoundsListener listener) {
        if (this.boundsListeners != null) {
            this.boundsListeners.remove(listener);
        }
    }

    public boolean isEmpty() {
        this.lock.readLock().lock();
        try {
            boolean bl = this.screenArea.isEmpty() || this.bounds.isEmpty();
            return bl;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    public ReferencedEnvelope getBounds() {
        this.lock.readLock().lock();
        try {
            ReferencedEnvelope referencedEnvelope = new ReferencedEnvelope(this.bounds);
            return referencedEnvelope;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    public void setBounds(ReferencedEnvelope requestedBounds) {
        this.lock.writeLock().lock();
        try {
            if (this.checkEditable("setBounds")) {
                ReferencedEnvelope old = this.bounds;
                this.copyBounds(requestedBounds);
                this.setTransforms(true);
                this.fireMapBoundsListenerMapBoundsChanged(MapBoundsEvent.Type.BOUNDS, old, this.bounds);
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    private void copyBounds(ReferencedEnvelope newBounds) {
        this.bounds = newBounds == null || newBounds.isEmpty() ? new ReferencedEnvelope() : new ReferencedEnvelope(newBounds);
    }

    public Rectangle getScreenArea() {
        this.lock.readLock().lock();
        try {
            Rectangle rectangle = new Rectangle(this.screenArea);
            return rectangle;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    public void setScreenArea(Rectangle screenArea) {
        this.lock.writeLock().lock();
        try {
            if (this.checkEditable("setScreenArea")) {
                this.doSetScreenArea(screenArea);
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    private void doSetScreenArea(Rectangle screenArea) {
        this.screenArea = screenArea == null || screenArea.isEmpty() ? new Rectangle() : new Rectangle(screenArea);
        if (this.fixedBoundsOnResize) {
            this.setTransforms(true);
        } else {
            this.setTransforms(false);
        }
    }

    public CoordinateReferenceSystem getCoordinateReferenceSystem() {
        this.lock.readLock().lock();
        try {
            CoordinateReferenceSystem coordinateReferenceSystem = this.bounds.getCoordinateReferenceSystem();
            return coordinateReferenceSystem;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    public void setCoordinateReferenceSystem(CoordinateReferenceSystem crs) {
        this.lock.writeLock().lock();
        try {
            if (this.checkEditable("setCoordinateReferenceSystem")) {
                if (crs == null) {
                    this.bounds = new ReferencedEnvelope((Envelope)this.bounds, null);
                } else if (!CRS.equalsIgnoreMetadata((Object)crs, (Object)this.bounds.getCoordinateReferenceSystem())) {
                    if (this.bounds.isEmpty()) {
                        this.bounds = new ReferencedEnvelope(crs);
                    } else {
                        try {
                            ReferencedEnvelope old = this.bounds;
                            this.bounds = this.bounds.transform(crs, true);
                            this.setTransforms(true);
                            this.fireMapBoundsListenerMapBoundsChanged(MapBoundsEvent.Type.CRS, old, this.bounds);
                        }
                        catch (Exception e) {
                            LOGGER.log(Level.FINE, "Difficulty transforming to {0}", crs);
                        }
                    }
                }
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    protected void fireMapBoundsListenerMapBoundsChanged(MapBoundsEvent.Type type, ReferencedEnvelope oldBounds, ReferencedEnvelope newBounds) {
        if (this.boundsListeners == null) {
            return;
        }
        if (newBounds == this.bounds) {
            newBounds = new ReferencedEnvelope(this.bounds);
        }
        MapBoundsEvent event = new MapBoundsEvent(this, type, oldBounds, newBounds);
        for (MapBoundsListener boundsListener : this.boundsListeners) {
            try {
                boundsListener.mapBoundsChanged(event);
            }
            catch (Throwable t) {
                if (!LOGGER.isLoggable(Level.FINER)) continue;
                LOGGER.logp(Level.FINE, boundsListener.getClass().getName(), "mapBoundsChanged", t.getLocalizedMessage(), t);
            }
        }
    }

    public AffineTransform getScreenToWorld() {
        this.lock.readLock().lock();
        try {
            AffineTransform affineTransform = this.screenToWorld == null ? null : new AffineTransform(this.screenToWorld);
            return affineTransform;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    public AffineTransform getWorldToScreen() {
        this.lock.readLock().lock();
        try {
            AffineTransform affineTransform = this.worldToScreen == null ? null : new AffineTransform(this.worldToScreen);
            return affineTransform;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    private void setTransforms(boolean newBounds) {
        if (this.screenArea.isEmpty()) {
            this.worldToScreen = null;
            this.screenToWorld = null;
            this.hasCenteringTransforms = false;
        } else if (this.bounds.isEmpty()) {
            this.screenToWorld = new AffineTransform();
            this.worldToScreen = new AffineTransform();
            this.hasCenteringTransforms = false;
        } else if (this.matchingAspectRatio) {
            if (newBounds || !this.hasCenteringTransforms) {
                this.calculateCenteringTransforms();
            }
            this.bounds = this.calculateActualBounds();
        } else {
            this.calculateSimpleTransforms(this.bounds);
            this.hasCenteringTransforms = false;
        }
    }

    private void calculateCenteringTransforms() {
        double xscale = this.screenArea.getWidth() / this.bounds.getWidth();
        double yscale = this.screenArea.getHeight() / this.bounds.getHeight();
        double scale = Math.min(xscale, yscale);
        double xoff = this.bounds.getMedian(0) * scale - this.screenArea.getCenterX();
        double yoff = this.bounds.getMedian(1) * scale + this.screenArea.getCenterY();
        this.worldToScreen = new AffineTransform(scale, 0.0, 0.0, -scale, -xoff, yoff);
        try {
            this.screenToWorld = this.worldToScreen.createInverse();
        }
        catch (NoninvertibleTransformException ex) {
            throw new RuntimeException("Unable to create coordinate transforms.", ex);
        }
        this.hasCenteringTransforms = true;
    }

    public boolean isFixedBoundsOnResize() {
        return this.fixedBoundsOnResize;
    }

    public void setFixedBoundsOnResize(boolean fixedBoundsOnResize) {
        this.fixedBoundsOnResize = fixedBoundsOnResize;
    }

    private void calculateSimpleTransforms(ReferencedEnvelope requestedBounds) {
        double xscale = this.screenArea.getWidth() / requestedBounds.getWidth();
        double yscale = this.screenArea.getHeight() / requestedBounds.getHeight();
        this.worldToScreen = new AffineTransform(xscale, 0.0, 0.0, -yscale, -xscale * requestedBounds.getMinX(), yscale * requestedBounds.getMaxY());
        try {
            this.screenToWorld = this.worldToScreen.createInverse();
        }
        catch (NoninvertibleTransformException ex) {
            throw new RuntimeException("Unable to create coordinate transforms.", ex);
        }
    }

    private ReferencedEnvelope calculateActualBounds() {
        Point2D.Double p0 = new Point2D.Double(this.screenArea.getMinX(), this.screenArea.getMinY());
        Point2D.Double p1 = new Point2D.Double(this.screenArea.getMaxX(), this.screenArea.getMaxY());
        this.screenToWorld.transform(p0, p0);
        this.screenToWorld.transform(p1, p1);
        return new ReferencedEnvelope(Math.min(((Point2D)p0).getX(), ((Point2D)p1).getX()), Math.max(((Point2D)p0).getX(), ((Point2D)p1).getX()), Math.min(((Point2D)p0).getY(), ((Point2D)p1).getY()), Math.max(((Point2D)p0).getY(), ((Point2D)p1).getY()), this.bounds.getCoordinateReferenceSystem());
    }

    private boolean checkEditable(String methodName) {
        boolean state = this.editable.get();
        if (!state && LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, "Ignored call to {0} because viewport is not editable", methodName);
        }
        return state;
    }
}

