/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 *
 *    (C) 2002-2008, Open Source Geospatial Foundation (OSGeo)
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 */
package org.geotools.renderer.style;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.font.FontRenderContext;
import java.awt.font.GlyphVector;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.WindowConstants;
import org.geotools.api.feature.Feature;
import org.geotools.api.filter.expression.Expression;
import org.geotools.api.style.ExternalMark;
import org.geotools.renderer.util.ExplicitBoundsShape;

/**
 * This factory accepts mark paths in the <code>ttf://fontName#code</code> format, where fontName is the name of a
 * TrueType font installed in the system, or a URL to a TTF file, and the code is the character code, which may be
 * expressed in decimal, hexadecimal (e.g. <code>0x10</code>) octal (e.g. <code>045</code>) form, as well as Unicode
 * codes (e.g. <code>U+F054</code> or <code>\uF054
 * </code>).
 *
 * @author Andrea Aime - TOPP
 */
public class TTFMarkFactory implements MarkFactory {

    private static FontRenderContext FONT_RENDER_CONTEXT = new FontRenderContext(new AffineTransform(), false, false);

    /** The factory is completely stateless, this single instance can be safely used across multiple threads */
    public static TTFMarkFactory INSTANCE = new TTFMarkFactory();

    @Override
    public Shape getShape(Graphics2D graphics, Expression symbolUrl, Feature feature) throws Exception {
        String markUrl = symbolUrl.evaluate(feature, String.class);

        // if it does not start with the right prefix, it's not our business
        if (!markUrl.startsWith("ttf://")) return null;

        // if it does not match the expected format, complain before exiting
        if (!markUrl.matches("ttf://.+#.+")) {
            throw new IllegalArgumentException("Mark URL font found, but does not match the required "
                    + "structure font://<fontName>#<charNumber>, e.g., ttf://wingdigs#0x7B. You specified "
                    + markUrl);
        }
        String[] fontElements = markUrl.substring(6).split("#");

        // get the symbol number
        String code = fontElements[1];
        char character;
        try {
            // see if a unicode escape sequence has been used
            if (code.startsWith("U+") || code.startsWith("\\u")) code = "0x" + code.substring(2);

            // this will handle most numeric formats like decimal, hex and octal
            character = (char) Integer.decode(code).intValue();
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException("Invalid character specification " + fontElements[1], e);
        }

        // look up the font
        String fontFamilyName = fontElements[0];
        return getShape(fontFamilyName, character);
    }

    private Shape getShape(String fontFamilyName, char character) {
        Font font = FontCache.getDefaultInstance().getFont(fontFamilyName);
        if (font == null) {
            throw new IllegalArgumentException("Unknown font " + fontFamilyName);
        }

        // handle charmap code reporting issues
        if (!font.canDisplay(character)) {
            char alternative = (char) (0xF000 | character);
            if (font.canDisplay(alternative)) {
                character = alternative;
            }
        }

        // build the shape out of the font
        GlyphVector textGlyphVector = font.createGlyphVector(FONT_RENDER_CONTEXT, new char[] {character});
        Shape s = textGlyphVector.getOutline();

        // have the shape be centered in the origin, and sitting in a square of side 1
        Rectangle2D bounds = s.getBounds2D();
        AffineTransform tx = new AffineTransform();
        double max = Math.max(bounds.getWidth(), bounds.getHeight());
        // all shapes are defined looking "upwards" (see ShapeMarkFactory or WellKnownMarkFactory)
        // but the fonts ones are flipped to compensate for the fact the y coords grow from top
        // to bottom on the screen. We have to flip the symbol so that it conforms to the
        // other marks convention
        tx.scale(1 / max, -1 / max);
        tx.translate(-bounds.getCenterX(), -bounds.getCenterY());
        ExplicitBoundsShape shape = new ExplicitBoundsShape(tx.createTransformedShape(s));
        shape.setBounds(new Rectangle2D.Double(-0.5, 0.5, 1.0, 1.0));
        return shape;
    }

    /** Returns a shape from an external mark definition */
    public Shape getShape(ExternalMark mark) {
        if (!"ttf".equals(mark.getFormat())) {
            return null;
        }

        String link = mark.getOnlineResource().getLinkage().toString();
        // if it does not start with the right prefix, it's not our business
        if (!link.startsWith("ttf://")) return null;

        String family;
        try {
            family = URLDecoder.decode(link.substring(6), "UTF-8");
        } catch (UnsupportedEncodingException e) {
            // UTF-8 not supported??
            throw new RuntimeException(e);
        }
        int markIdx = mark.getMarkIndex();
        char character = (char) markIdx;
        return getShape(family, character);
    }

    @SuppressWarnings("PMD.SystemPrintln")
    public static void main(String[] args) {
        BufferedImage image = new BufferedImage(300, 300, BufferedImage.TYPE_4BYTE_ABGR);
        Graphics2D g2d = (Graphics2D) image.getGraphics();
        g2d.setColor(Color.BLACK);

        char c = 0xF041;
        System.out.println((int) c);

        Font font = new Font("Webdings", Font.PLAIN, 60);
        for (int i = 0; i < 65536; i++) if (font.canDisplay(i)) System.out.println(i + ": " + Long.toHexString(i));
        GlyphVector textGlyphVector = font.createGlyphVector(FONT_RENDER_CONTEXT, new char[] {,});
        Shape shape = textGlyphVector.getOutline();
        g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        g2d.translate(150, 150);
        g2d.setColor(Color.BLUE);
        g2d.fill(shape);

        g2d.setColor(Color.BLACK);
        g2d.setFont(font);
        g2d.drawString(new String(new char[] {c}), 0, 50);

        g2d.dispose();
        JFrame frame = new JFrame("Test");
        frame.setContentPane(new JLabel(new ImageIcon(image)));
        frame.pack();
        frame.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
        frame.setVisible(true);
    }
}
