/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.tpk;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.tpk.TPKReader;
import org.geotools.tpk.TPKTile;
import org.geotools.tpk.TPKZoomLevel;
import org.geotools.tpk.TPKZoomLevelV1;
import org.geotools.tpk.TPKZoomLevelV2;
import org.geotools.util.logging.Logging;
import org.opengis.geometry.Envelope;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class TPKFile {
    private static final Logger LOGGER = Logging.getLogger(TPKFile.class);
    private static double WORLD_CIRCUMFERENCE = 4.007501669E7;
    private static double ORIGIN_OFFSET = WORLD_CIRCUMFERENCE / 2.0;
    private static long TILE_PIXEL_SIZE = 256L;
    private static String CONFIGURATION_FILE = "/conf.xml";
    private static String TAG_STORAGE_FORMAT = "StorageFormat";
    private static String TAG_LOD_INFO = "LODInfo";
    private static String TAG_TILE_FORMAT = "CacheTileFormat";
    private static String TAG_LEVEL_ID = "LevelID";
    private static String TAG_RESOLUTION = "Resolution";
    private static String COMPACT_CACHE_V1 = "esriMapCacheStorageModeCompact";
    private static String COMPACT_CACHE_V2 = "esriMapCacheStorageModeCompactV2";
    private static String LEVEL_FOLDER = "_alllayers/L%02d/";
    private static String BUNDLE_DATA_EXTENSION = ".bundle";
    private static String BUNDLE_INDEX_EXTENSION = ".bundlx";
    private ZipFile theTPK;
    private Map<String, ZipEntry> zipEntryMap = new HashMap<String, ZipEntry>();
    private Map<Long, TPKZoomLevel> zoomLevelMap;
    private String imageFormat;
    private Envelope bounds;
    private Map<Long, Long> zoomLevelMapping;
    private Map<Long, Double> zoomLevelResolutionMap;
    private CacheType cacheType;

    public TPKFile(File theFile, Map<Long, TPKZoomLevel> zoomLevelMap) {
        this.openTPK(theFile);
        this.zoomLevelMap = zoomLevelMap;
        String xmlConf = this.zipEntryMap.keySet().stream().filter(s -> s.endsWith(CONFIGURATION_FILE)).findFirst().orElse(null);
        this.parseConfigurationFile(this.zipEntryMap.get(xmlConf));
        this.loadZoomLevels();
    }

    public TPKFile(File theFile, Map<Long, TPKZoomLevel> zoomLevelMap, Envelope bounds, String imageFormat) {
        this.openTPK(theFile);
        this.imageFormat = imageFormat;
        this.bounds = bounds;
        this.zoomLevelMap = zoomLevelMap;
        zoomLevelMap.values().forEach(zl -> zl.setTPKandEntryMap(this.theTPK, this.zipEntryMap));
    }

    public void close() {
        this.zoomLevelMap.values().forEach(TPKZoomLevel::releaseResources);
        this.zipEntryMap.clear();
        try {
            this.theTPK.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public long getMinZoomLevel() {
        return (Long)this.zoomLevelMap.keySet().stream().min(Long::compare).get();
    }

    public long getMaxZoomLevel() {
        return (Long)this.zoomLevelMap.keySet().stream().max(Long::compare).get();
    }

    public String getImageFormat() {
        return this.imageFormat;
    }

    public Envelope getBounds() {
        if (this.bounds == null) {
            if (this.zoomLevelMap != null && this.zoomLevelMap.size() > 0) {
                this.calculateBoundsFromTileset(this.zoomLevelMap.get(this.getMaxZoomLevel()));
            } else {
                throw new RuntimeException("Can't get bounds, zoomLevelMap not initialized");
            }
        }
        return this.bounds;
    }

    public long getClosestZoom(long zoomLevel) {
        Set<Long> zooms = this.zoomLevelMap.keySet();
        if (zooms.contains(zoomLevel)) {
            return zoomLevel;
        }
        long smallestDiff = Long.MAX_VALUE;
        long closestZoom = 0L;
        for (long zl : zooms) {
            long diff = Math.abs(zoomLevel - zl);
            if (diff >= smallestDiff) continue;
            smallestDiff = diff;
            closestZoom = zl;
        }
        return closestZoom;
    }

    public long getMinColumn(long zoomLevel) {
        long retValue = -1L;
        if (this.zoomLevelMap.containsKey(zoomLevel)) {
            retValue = this.zoomLevelMap.get(zoomLevel).getMinColumn();
        }
        return retValue;
    }

    public long getMaxColumn(long zoomLevel) {
        long retValue = -1L;
        if (this.zoomLevelMap.containsKey(zoomLevel)) {
            retValue = this.zoomLevelMap.get(zoomLevel).getMaxColumn();
        }
        return retValue;
    }

    public long getMinRow(long zoomLevel) {
        long retValue = -1L;
        if (this.zoomLevelMap.containsKey(zoomLevel)) {
            retValue = this.zoomLevelMap.get(zoomLevel).getMinRow();
        }
        return retValue;
    }

    public long getMaxRow(long zoomLevel) {
        long retValue = -1L;
        if (this.zoomLevelMap.containsKey(zoomLevel)) {
            retValue = this.zoomLevelMap.get(zoomLevel).getMaxRow();
        }
        return retValue;
    }

    public List<TPKTile> getTiles(long zoomLevel, long top, long bottom, long left, long right, String format) {
        if (this.zoomLevelMap.containsKey(zoomLevel)) {
            return this.zoomLevelMap.get(zoomLevel).getTiles(top, bottom, left, right, format);
        }
        return null;
    }

    private void openTPK(File theFile) {
        try {
            this.theTPK = new ZipFile(theFile);
        }
        catch (IOException ex) {
            throw new RuntimeException("Unable to open TPK file", ex);
        }
        Enumeration<? extends ZipEntry> zipEntries = this.theTPK.entries();
        while (zipEntries.hasMoreElements()) {
            ZipEntry entry = zipEntries.nextElement();
            this.zipEntryMap.put(entry.getName(), entry);
        }
    }

    private void parseConfigurationFile(ZipEntry confFile) {
        this.zoomLevelMapping = new HashMap<Long, Long>();
        this.zoomLevelResolutionMap = new HashMap<Long, Double>();
        try (InputStream is = this.theTPK.getInputStream(confFile);){
            NodeList ctfs;
            Node ctf;
            DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();
            Document confDoc = dBuilder.parse(is);
            NodeList sfs = confDoc.getElementsByTagName(TAG_STORAGE_FORMAT);
            Node sf = sfs.item(0);
            if (sf.getNodeType() == 1) {
                String storageFormat = sf.getTextContent();
                if (storageFormat.equals(COMPACT_CACHE_V1)) {
                    this.cacheType = CacheType.V1;
                } else if (storageFormat.equals(COMPACT_CACHE_V2)) {
                    this.cacheType = CacheType.V2;
                } else {
                    throw new RuntimeException("Unknown value for StorageFormat element");
                }
            }
            if ((ctf = (ctfs = confDoc.getElementsByTagName(TAG_TILE_FORMAT)).item(0)).getNodeType() == 1) {
                this.imageFormat = ctf.getTextContent();
            }
            NodeList lods = confDoc.getElementsByTagName(TAG_LOD_INFO);
            for (int i = 0; i < lods.getLength(); ++i) {
                this.parseLodInfo(lods.item(i));
            }
        }
        catch (Exception ex) {
            throw new RuntimeException("Caught exception opening/processing conf.xml", ex);
        }
    }

    private void parseLodInfo(Node lodInfo) {
        if (lodInfo.getNodeType() == 1) {
            Element lod = (Element)lodInfo;
            NodeList lidList = lod.getElementsByTagName(TAG_LEVEL_ID);
            NodeList resList = lod.getElementsByTagName(TAG_RESOLUTION);
            Node lid = lidList.item(0);
            Node res = resList.item(0);
            if (lid.getNodeType() == 1 && res.getNodeType() == 1) {
                String levelString = lid.getTextContent();
                Long level = Long.valueOf(levelString);
                String resString = res.getTextContent();
                Double resolution = Double.valueOf(resString);
                long zoom_level = Math.round(this.log2(WORLD_CIRCUMFERENCE / (resolution * (double)TILE_PIXEL_SIZE)));
                this.zoomLevelMapping.put(level, zoom_level);
                this.zoomLevelResolutionMap.put(zoom_level, resolution);
            }
        }
    }

    private double log2(double number) {
        return Math.log(number) / Math.log(2.0);
    }

    private void loadZoomLevels() {
        long startLoad = System.currentTimeMillis();
        for (Long levelId : this.zoomLevelMapping.keySet()) {
            String levelFolder = String.format(LEVEL_FOLDER, levelId);
            List<String> indexes = null;
            List<String> bundles = this.zipEntryMap.keySet().stream().filter(s -> s.contains(levelFolder)).filter(s -> s.endsWith(BUNDLE_DATA_EXTENSION)).collect(Collectors.toList());
            if (this.cacheType == CacheType.V1) {
                indexes = this.zipEntryMap.keySet().stream().filter(s -> s.contains(levelFolder)).filter(s -> s.endsWith(BUNDLE_INDEX_EXTENSION)).collect(Collectors.toList());
            }
            if (bundles.isEmpty()) continue;
            Long zoomLevel = this.zoomLevelMapping.get(levelId);
            TPKZoomLevel zlObj = null;
            if (this.cacheType == CacheType.V1) {
                zlObj = new TPKZoomLevelV1(this.theTPK, this.zipEntryMap, bundles, indexes, zoomLevel);
            } else if (this.cacheType == CacheType.V2) {
                zlObj = new TPKZoomLevelV2(this.theTPK, this.zipEntryMap, bundles, zoomLevel);
            }
            this.zoomLevelMap.put(zoomLevel, zlObj);
        }
        String msg = String.format("Loaded zoom levels in %d milliseconds", System.currentTimeMillis() - startLoad);
        LOGGER.fine(msg);
    }

    private void calculateBoundsFromTileset(TPKZoomLevel zl) {
        double resolution = this.zoomLevelResolutionMap.get(zl.getZoomLevel());
        double minX = ((double)(zl.getMinColumn() * TILE_PIXEL_SIZE) * resolution - ORIGIN_OFFSET) / ORIGIN_OFFSET * 180.0;
        double maxX = ((double)((zl.getMaxColumn() + 1L) * TILE_PIXEL_SIZE) * resolution - ORIGIN_OFFSET) / ORIGIN_OFFSET * 180.0;
        double lat = ((double)(zl.getMinRow() * TILE_PIXEL_SIZE) * resolution - ORIGIN_OFFSET) / ORIGIN_OFFSET * 180.0;
        double minY = 57.29577951308232 * (2.0 * Math.atan(Math.exp(lat * Math.PI / 180.0)) - 1.5707963267948966);
        lat = ((double)((zl.getMaxRow() + 1L) * TILE_PIXEL_SIZE) * resolution - ORIGIN_OFFSET) / ORIGIN_OFFSET * 180.0;
        double maxY = 57.29577951308232 * (2.0 * Math.atan(Math.exp(lat * Math.PI / 180.0)) - 1.5707963267948966);
        this.bounds = new ReferencedEnvelope(minX, maxX, minY, maxY, TPKReader.WGS_84);
    }

    private static enum CacheType {
        V1,
        V2;

    }
}

