/*
 * Decompiled with CFR 0.152.
 */
package org.geowebcache.diskquota.bdb;

import com.google.common.base.Objects;
import com.sleepycat.je.CursorConfig;
import com.sleepycat.je.Environment;
import com.sleepycat.je.LockMode;
import com.sleepycat.je.Transaction;
import com.sleepycat.persist.EntityCursor;
import com.sleepycat.persist.EntityStore;
import com.sleepycat.persist.PrimaryIndex;
import com.sleepycat.persist.SecondaryIndex;
import java.io.File;
import java.io.IOException;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.geotools.util.logging.Logging;
import org.geowebcache.config.ConfigurationException;
import org.geowebcache.diskquota.QuotaStore;
import org.geowebcache.diskquota.bdb.EntityStoreBuilder;
import org.geowebcache.diskquota.storage.PageStats;
import org.geowebcache.diskquota.storage.PageStatsPayload;
import org.geowebcache.diskquota.storage.PageStoreConfig;
import org.geowebcache.diskquota.storage.Quota;
import org.geowebcache.diskquota.storage.TilePage;
import org.geowebcache.diskquota.storage.TilePageCalculator;
import org.geowebcache.diskquota.storage.TileSet;
import org.geowebcache.diskquota.storage.TileSetVisitor;
import org.geowebcache.storage.DefaultStorageFinder;
import org.geowebcache.util.FileUtils;
import org.springframework.scheduling.concurrent.CustomizableThreadFactory;
import org.springframework.util.Assert;

public class BDBQuotaStore
implements QuotaStore {
    private static final Logger log = Logging.getLogger((String)BDBQuotaStore.class.getName());
    private static final String GLOBAL_QUOTA_NAME = "___GLOBAL_QUOTA___";
    public static final String STORE_VERSION = "1.1";
    private static final String VERSION_FILE = "version.txt";
    private EntityStore entityStore;
    private final String cacheRootDir;
    private final TilePageCalculator tilePageCalculator;
    private static ExecutorService transactionRunner;
    private PrimaryIndex<String, TileSet> tileSetById;
    private PrimaryIndex<Integer, Quota> usedQuotaById;
    private PrimaryIndex<Long, TilePage> pageById;
    private PrimaryIndex<Long, PageStats> pageStatsById;
    private SecondaryIndex<String, String, TileSet> tileSetsByLayer;
    private SecondaryIndex<String, Long, TilePage> pageByKey;
    private SecondaryIndex<String, Long, TilePage> pagesByTileSetId;
    private SecondaryIndex<Long, Long, PageStats> pageStatsByPageId;
    private SecondaryIndex<Float, Long, PageStats> pageStatsByLRU;
    private SecondaryIndex<Float, Long, PageStats> pageStatsByLFU;
    private SecondaryIndex<String, Integer, Quota> usedQuotaByTileSetId;
    private volatile boolean open;
    private boolean diskQuotaEnabled;

    public BDBQuotaStore(DefaultStorageFinder cacheDirFinder, TilePageCalculator tilePageCalculator) throws ConfigurationException {
        Assert.notNull((Object)cacheDirFinder, (String)"cacheDirFinder can't be null");
        Assert.notNull((Object)tilePageCalculator, (String)"tilePageCalculator can't be null");
        this.tilePageCalculator = tilePageCalculator;
        this.cacheRootDir = cacheDirFinder.getDefaultPath();
        boolean disabled = Boolean.parseBoolean(cacheDirFinder.findEnvVar("GWC_DISKQUOTA_DISABLED"));
        if (disabled) {
            log.warning(" -- Found environment variable GWC_DISKQUOTA_DISABLED set to true. DiskQuotaMonitor is disabled.");
        }
        this.diskQuotaEnabled = !disabled;
    }

    public void startUp() throws InterruptedException, IOException {
        if (!this.diskQuotaEnabled) {
            log.info(this.getClass().getName() + " won't start, got env variable GWC_DISKQUOTA_DISABLED=true");
            return;
        }
        this.open = true;
        File storeDirectory = new File(this.cacheRootDir, "diskquota_page_store");
        storeDirectory.mkdirs();
        File version = new File(storeDirectory, VERSION_FILE);
        if (FileUtils.listFilesNullSafe((File)storeDirectory).length == 0) {
            try {
                org.apache.commons.io.FileUtils.write((File)version, (CharSequence)STORE_VERSION, (String)"UTF-8");
            }
            catch (IOException e) {
                throw new IOException("BDB DiskQuota could not write version.txt to new database", e);
            }
        }
        try {
            String versionString = org.apache.commons.io.FileUtils.readFileToString((File)version, (String)"UTF-8");
            if (!versionString.equals(STORE_VERSION)) {
                throw new IOException("BDB DiskQuota does not support database version " + versionString);
            }
        }
        catch (IOException e) {
            throw new IOException("BDB DiskQuota could not read version.txt to detemine database version", e);
        }
        CustomizableThreadFactory tf = new CustomizableThreadFactory("GWC DiskQuota Store Writer-");
        transactionRunner = Executors.newFixedThreadPool(1, (ThreadFactory)tf);
        try {
            this.configure(storeDirectory);
            this.deleteStaleLayersAndCreateMissingTileSets();
            log.config("Berkeley DB JE Disk Quota page store configured at " + storeDirectory.getAbsolutePath());
        }
        catch (RuntimeException e) {
            transactionRunner.shutdownNow();
            throw e;
        }
        log.config("Quota Store initialized. Global quota: " + this.getGloballyUsedQuota().toNiceString());
    }

    public void close() throws Exception {
        if (!this.diskQuotaEnabled) {
            return;
        }
        this.open = false;
        log.config("Requesting to close quota store...");
        transactionRunner.shutdown();
        try {
            transactionRunner.awaitTermination(30000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException ie) {
            log.log(Level.SEVERE, "Time out shutting down quota store write thread, trying to close the entity store as is.", ie);
            Thread.currentThread().interrupt();
        }
        finally {
            Environment environment = this.entityStore.getEnvironment();
            this.entityStore.close();
            environment.close();
        }
        log.config("Quota store closed.");
    }

    private void configure(File storeDirectory) throws InterruptedException {
        EntityStore entityStore;
        PageStoreConfig config = new PageStoreConfig();
        EntityStoreBuilder builder = new EntityStoreBuilder(config);
        this.entityStore = entityStore = builder.buildEntityStore(storeDirectory, null);
        this.tileSetById = entityStore.getPrimaryIndex(String.class, TileSet.class);
        this.pageById = entityStore.getPrimaryIndex(Long.class, TilePage.class);
        this.pageStatsById = entityStore.getPrimaryIndex(Long.class, PageStats.class);
        this.usedQuotaById = entityStore.getPrimaryIndex(Integer.class, Quota.class);
        this.pageByKey = entityStore.getSecondaryIndex(this.pageById, String.class, "page_key");
        this.pagesByTileSetId = entityStore.getSecondaryIndex(this.pageById, String.class, "tileset_id_fk");
        this.tileSetsByLayer = entityStore.getSecondaryIndex(this.tileSetById, String.class, "layer");
        this.pageStatsByLRU = entityStore.getSecondaryIndex(this.pageStatsById, Float.class, "LRU");
        this.pageStatsByLFU = entityStore.getSecondaryIndex(this.pageStatsById, Float.class, "LFU");
        this.usedQuotaByTileSetId = entityStore.getSecondaryIndex(this.usedQuotaById, String.class, "tileset_id");
        this.pageStatsByPageId = entityStore.getSecondaryIndex(this.pageStatsById, Long.class, "page_stats_by_page_id");
    }

    public void createLayer(String layerName) throws InterruptedException {
        this.issueSync(() -> {
            Transaction transaction = this.entityStore.getEnvironment().beginTransaction(null, null);
            try {
                this.createLayer(layerName, transaction);
                transaction.commit();
            }
            catch (RuntimeException e) {
                transaction.abort();
            }
            return null;
        });
    }

    private void createLayer(String layerName, Transaction transaction) {
        Set layerTileSets = this.tilePageCalculator.getTileSetsFor(layerName);
        for (TileSet tset : layerTileSets) {
            this.getOrCreateTileSet(transaction, tset);
        }
    }

    private TileSet getOrCreateTileSet(Transaction transaction, TileSet tset) {
        String id = tset.getId();
        TileSet stored = (TileSet)this.tileSetById.get(transaction, (Object)id, LockMode.DEFAULT);
        if (null == stored) {
            log.fine("Creating TileSet for quota tracking: " + String.valueOf(tset));
            this.tileSetById.putNoReturn(transaction, (Object)tset);
            stored = tset;
            Quota tileSetUsedQuota = new Quota();
            tileSetUsedQuota.setTileSetId(tset.getId());
            this.usedQuotaById.putNoReturn(transaction, (Object)tileSetUsedQuota);
        }
        return stored;
    }

    private <E> Future<E> issue(Callable<E> command) {
        if (!this.open) {
            throw new IllegalStateException("QuotaStore is closed.");
        }
        Future<E> future = transactionRunner.submit(command);
        return future;
    }

    private <E> E issueSync(Callable<E> command) throws InterruptedException {
        Future<E> result = this.issue(command);
        try {
            return result.get();
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (InterruptedException e) {
            log.fine("Caught InterruptedException while waiting for command " + command.getClass().getSimpleName());
            throw e;
        }
        catch (ExecutionException e) {
            log.log(Level.WARNING, e.getMessage(), e);
            Throwable cause = e.getCause();
            if (cause instanceof RuntimeException) {
                throw (RuntimeException)cause;
            }
            throw new RuntimeException(cause);
        }
    }

    private void deleteStaleLayersAndCreateMissingTileSets() throws InterruptedException {
        this.issueSync(new StartUpInitializer());
    }

    public Quota getGloballyUsedQuota() throws InterruptedException {
        return this.getUsedQuotaByTileSetId(GLOBAL_QUOTA_NAME);
    }

    public Quota getUsedQuotaByTileSetId(String tileSetId) throws InterruptedException {
        Quota usedQuota = this.issueSync(new UsedQuotaByTileSetId(tileSetId));
        return usedQuota;
    }

    public void deleteLayer(String layerName) {
        Assert.notNull((Object)layerName, (String)"LayerName must be non null");
        this.issue(new Deleter(layerName, ts -> true));
    }

    public void deleteGridSubset(String layerName, String gridSetId) {
        this.issue(new Deleter(layerName, ts -> Objects.equal((Object)ts.getGridsetId(), (Object)gridSetId)));
    }

    public void deleteParameters(String layerName, String parametersId) {
        this.issue(new Deleter(layerName, ts -> Objects.equal((Object)ts.getParametersId(), (Object)parametersId)));
    }

    public void renameLayer(String oldLayerName, String newLayerName) throws InterruptedException {
        Assert.notNull((Object)oldLayerName, (String)"Old layer name must be non null");
        Assert.notNull((Object)newLayerName, (String)"New layer name must be non null");
        this.issueSync(new RenameLayer(oldLayerName, newLayerName));
    }

    public Quota getUsedQuotaByLayerName(String layerName) throws InterruptedException {
        return this.issueSync(new UsedQuotaByLayerName(layerName));
    }

    public long[][] getTilesForPage(TilePage page) throws InterruptedException {
        TileSet tileSet = this.getTileSetById(page.getTileSetId());
        long[][] gridCoverage = this.tilePageCalculator.toGridCoverage(tileSet, page);
        return gridCoverage;
    }

    public Set<TileSet> getTileSets() {
        HashMap map = new HashMap(this.tileSetById.map());
        map.remove(GLOBAL_QUOTA_NAME);
        HashSet<TileSet> hashSet = new HashSet<TileSet>(map.values());
        return hashSet;
    }

    public TileSet getTileSetById(String tileSetId) throws InterruptedException {
        return this.issueSync(() -> {
            TileSet tileSet = (TileSet)this.tileSetById.get((Object)tileSetId);
            if (tileSet == null) {
                throw new IllegalArgumentException("TileSet does not exist: " + tileSetId);
            }
            return tileSet;
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void accept(TileSetVisitor visitor) {
        try (EntityCursor cursor = this.tileSetById.entities();){
            TileSet tileSet;
            while ((tileSet = (TileSet)cursor.next()) != null) {
                visitor.visit(tileSet, (QuotaStore)this);
            }
        }
    }

    public TilePageCalculator getTilePageCalculator() {
        return this.tilePageCalculator;
    }

    public void addToQuotaAndTileCounts(TileSet tileSet, Quota quotaDiff, Collection<PageStatsPayload> tileCountDiffs) throws InterruptedException {
        this.issueSync(new AddToQuotaAndTileCounts(tileSet, quotaDiff, tileCountDiffs));
    }

    public Future<List<PageStats>> addHitsAndSetAccesTime(Collection<PageStatsPayload> statsUpdates) {
        Assert.notNull(statsUpdates, (String)"Stats update must be non null");
        return this.issue(new AddHitsAndSetAccesTime(statsUpdates));
    }

    public TilePage getLeastFrequentlyUsedPage(Set<String> layerNames) throws InterruptedException {
        SecondaryIndex<Float, Long, PageStats> expirationPolicyIndex = this.pageStatsByLFU;
        TilePage nextToExpire = this.issueSync(new FindPageToExpireByLayer(expirationPolicyIndex, layerNames));
        return nextToExpire;
    }

    public TilePage getLeastRecentlyUsedPage(Set<String> layerNames) throws InterruptedException {
        SecondaryIndex<Float, Long, PageStats> expirationPolicyIndex = this.pageStatsByLRU;
        TilePage nextToExpire = this.issueSync(new FindPageToExpireByLayer(expirationPolicyIndex, layerNames));
        return nextToExpire;
    }

    public PageStats setTruncated(TilePage tilePage) throws InterruptedException {
        return this.issueSync(new TruncatePage(tilePage));
    }

    private class TruncatePage
    implements Callable<PageStats> {
        private final TilePage tilePage;

        public TruncatePage(TilePage tilePage) {
            this.tilePage = tilePage;
        }

        @Override
        public PageStats call() throws Exception {
            Transaction tx = BDBQuotaStore.this.entityStore.getEnvironment().beginTransaction(null, null);
            try {
                PageStats pageStats = (PageStats)BDBQuotaStore.this.pageStatsByPageId.get(tx, (Object)this.tilePage.getId(), null);
                if (pageStats != null) {
                    pageStats.setFillFactor(0.0f);
                    BDBQuotaStore.this.pageStatsById.putNoReturn(tx, (Object)pageStats);
                }
                tx.commit();
                return pageStats;
            }
            catch (Exception e) {
                tx.abort();
                throw e;
            }
        }
    }

    private class FindPageToExpireByLayer
    implements Callable<TilePage> {
        private final SecondaryIndex<Float, Long, PageStats> expirationPolicyIndex;
        private final Set<String> layerNames;

        public FindPageToExpireByLayer(SecondaryIndex<Float, Long, PageStats> expirationPolicyIndex, Set<String> layerNames) {
            this.expirationPolicyIndex = expirationPolicyIndex;
            this.layerNames = layerNames;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public TilePage call() throws Exception {
            HashSet<String> tileSetIds = new HashSet<String>();
            for (String layerName : this.layerNames) {
                try (EntityCursor keys = BDBQuotaStore.this.tileSetsByLayer.entities((Object)layerName, true, (Object)layerName, true);){
                    TileSet tileSet;
                    while ((tileSet = (TileSet)keys.next()) != null) {
                        tileSetIds.add(tileSet.getId());
                    }
                }
            }
            TilePage nextToExpire = null;
            try (EntityCursor pageStatsCursor = this.expirationPolicyIndex.entities();){
                PageStats pageStats;
                while ((pageStats = (PageStats)pageStatsCursor.next()) != null) {
                    long pageId;
                    TilePage tilePage;
                    String tileSetId;
                    if (!(pageStats.getFillFactor() > 0.0f) || !tileSetIds.contains(tileSetId = (tilePage = (TilePage)BDBQuotaStore.this.pageById.get((Object)(pageId = pageStats.getPageId()))).getTileSetId())) continue;
                    nextToExpire = tilePage;
                    break;
                }
            }
            return nextToExpire;
        }
    }

    private class AddHitsAndSetAccesTime
    implements Callable<List<PageStats>> {
        private final Collection<PageStatsPayload> statsUpdates;

        public AddHitsAndSetAccesTime(Collection<PageStatsPayload> statsUpdates) {
            this.statsUpdates = statsUpdates;
        }

        @Override
        public List<PageStats> call() throws Exception {
            ArrayList<PageStats> allStats = new ArrayList<PageStats>(this.statsUpdates.size());
            PageStats pageStats = null;
            Transaction tx = BDBQuotaStore.this.entityStore.getEnvironment().beginTransaction(null, null);
            try {
                for (PageStatsPayload payload : this.statsUpdates) {
                    TilePage page = payload.getPage();
                    TileSet storedTileset = (TileSet)BDBQuotaStore.this.tileSetById.get(tx, (Object)page.getTileSetId(), LockMode.DEFAULT);
                    if (null == storedTileset) {
                        log.info("Can't add usage stats. TileSet does not exist. Was it deleted? " + page.getTileSetId());
                        continue;
                    }
                    TilePage storedPage = (TilePage)BDBQuotaStore.this.pageByKey.get(tx, (Object)page.getKey(), null);
                    if (storedPage == null) {
                        BDBQuotaStore.this.pageById.put(tx, (Object)page);
                        storedPage = page;
                        pageStats = new PageStats(storedPage.getId());
                    } else {
                        pageStats = (PageStats)BDBQuotaStore.this.pageStatsByPageId.get(tx, (Object)storedPage.getId(), null);
                    }
                    int addedHits = payload.getNumHits();
                    int lastAccessTimeMinutes = (int)(payload.getLastAccessTime() / 1000L / 60L);
                    int creationTimeMinutes = storedPage.getCreationTimeMinutes();
                    pageStats.addHitsAndAccessTime((long)addedHits, lastAccessTimeMinutes, creationTimeMinutes);
                    BDBQuotaStore.this.pageStatsById.putNoReturn(tx, (Object)pageStats);
                    allStats.add(pageStats);
                }
                tx.commit();
                return allStats;
            }
            catch (RuntimeException e) {
                tx.abort();
                throw e;
            }
        }
    }

    private class AddToQuotaAndTileCounts
    implements Callable<Void> {
        private final TileSet tileSet;
        private final Collection<PageStatsPayload> tileCountDiffs;
        private final Quota quotaDiff;

        public AddToQuotaAndTileCounts(TileSet tileSet, Quota quotaDiff, Collection<PageStatsPayload> tileCountDiffs) {
            this.tileSet = tileSet;
            this.quotaDiff = quotaDiff;
            this.tileCountDiffs = tileCountDiffs;
        }

        @Override
        public Void call() throws Exception {
            Transaction tx = BDBQuotaStore.this.entityStore.getEnvironment().beginTransaction(null, null);
            try {
                TileSet storedTileset = BDBQuotaStore.this.getOrCreateTileSet(tx, this.tileSet);
                this.addToUsedQuota(tx, storedTileset, this.quotaDiff);
                if (!this.tileCountDiffs.isEmpty()) {
                    for (PageStatsPayload payload : this.tileCountDiffs) {
                        PageStats pageStats;
                        TilePage page = payload.getPage();
                        String pageKey = page.getKey();
                        TilePage storedPage = (TilePage)BDBQuotaStore.this.pageByKey.get(tx, (Object)pageKey, LockMode.DEFAULT);
                        if (null == storedPage) {
                            BDBQuotaStore.this.pageById.put(tx, (Object)page);
                            storedPage = page;
                            pageStats = new PageStats(storedPage.getId());
                        } else {
                            pageStats = (PageStats)BDBQuotaStore.this.pageStatsByPageId.get(tx, (Object)storedPage.getId(), null);
                        }
                        byte level = page.getZoomLevel();
                        BigInteger tilesPerPage = BDBQuotaStore.this.tilePageCalculator.getTilesPerPage(this.tileSet, (int)level);
                        int tilesAdded = payload.getNumTiles();
                        pageStats.addTiles((long)tilesAdded, tilesPerPage);
                        BDBQuotaStore.this.pageStatsById.putNoReturn(tx, (Object)pageStats);
                    }
                }
                tx.commit();
                return null;
            }
            catch (RuntimeException e) {
                tx.abort();
                throw e;
            }
        }

        private void addToUsedQuota(Transaction tx, TileSet tileSet, Quota quotaDiff) {
            Quota usedQuota = (Quota)BDBQuotaStore.this.usedQuotaByTileSetId.get(tx, (Object)tileSet.getId(), LockMode.DEFAULT);
            Quota globalQuota = (Quota)BDBQuotaStore.this.usedQuotaByTileSetId.get(tx, (Object)BDBQuotaStore.GLOBAL_QUOTA_NAME, LockMode.DEFAULT);
            usedQuota.add(quotaDiff);
            globalQuota.add(quotaDiff);
            BDBQuotaStore.this.usedQuotaById.putNoReturn(tx, (Object)usedQuota);
            BDBQuotaStore.this.usedQuotaById.putNoReturn(tx, (Object)globalQuota);
        }
    }

    private final class UsedQuotaByLayerName
    implements Callable<Quota> {
        private final String layerName;

        public UsedQuotaByLayerName(String layerName) {
            this.layerName = layerName;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public Quota call() throws Exception {
            Quota aggregated = null;
            try (EntityCursor layerTileSetsIds = BDBQuotaStore.this.tileSetsByLayer.entities(null, (Object)this.layerName, true, (Object)this.layerName, true, CursorConfig.DEFAULT);){
                TileSet tileSet;
                while (null != (tileSet = (TileSet)layerTileSetsIds.next())) {
                    if (aggregated == null) {
                        aggregated = new Quota();
                    }
                    Quota tileSetUsedQuota = new UsedQuotaByTileSetId(tileSet.getId()).call();
                    aggregated.add(tileSetUsedQuota);
                }
            }
            if (aggregated == null) {
                aggregated = new Quota();
            }
            return aggregated;
        }
    }

    private class RenameLayer
    implements Callable<Void> {
        private final String oldLayerName;
        private final String newLayerName;

        public RenameLayer(String oldLayerName, String newLayerName) {
            this.oldLayerName = oldLayerName;
            this.newLayerName = newLayerName;
        }

        @Override
        public Void call() throws Exception {
            Transaction transaction = BDBQuotaStore.this.entityStore.getEnvironment().beginTransaction(null, null);
            try {
                this.copyTileSets(transaction);
                Deleter deleteCommand = new Deleter(this.oldLayerName, ts -> true);
                deleteCommand.call(transaction);
                transaction.commit();
            }
            catch (RuntimeException e) {
                transaction.abort();
                throw e;
            }
            return null;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void copyTileSets(Transaction transaction) {
            try (EntityCursor tileSets = BDBQuotaStore.this.tileSetsByLayer.entities(transaction, (Object)this.oldLayerName, true, (Object)this.oldLayerName, true, null);){
                TileSet oldTileSet;
                while (null != (oldTileSet = (TileSet)tileSets.next())) {
                    String gridsetId = oldTileSet.getGridsetId();
                    String blobFormat = oldTileSet.getBlobFormat();
                    String parametersId = oldTileSet.getParametersId();
                    TileSet newTileSet = new TileSet(this.newLayerName, gridsetId, blobFormat, parametersId);
                    newTileSet = BDBQuotaStore.this.getOrCreateTileSet(transaction, newTileSet);
                    String oldTileSetId = oldTileSet.getId();
                    String newTileSetId = newTileSet.getId();
                    Quota oldQuota = (Quota)BDBQuotaStore.this.usedQuotaByTileSetId.get(transaction, (Object)oldTileSetId, LockMode.DEFAULT);
                    Quota newQuota = (Quota)BDBQuotaStore.this.usedQuotaByTileSetId.get(transaction, (Object)newTileSetId, LockMode.DEFAULT);
                    newQuota.setBytes(oldQuota.getBytes());
                    BDBQuotaStore.this.usedQuotaById.putNoReturn(transaction, (Object)newQuota);
                    try (EntityCursor oldPages = BDBQuotaStore.this.pagesByTileSetId.entities(transaction, (Object)oldTileSetId, true, (Object)oldTileSetId, true, CursorConfig.DEFAULT);){
                        TilePage oldPage;
                        while (null != (oldPage = (TilePage)oldPages.next())) {
                            long oldPageId = oldPage.getId();
                            TilePage newPage = new TilePage(newTileSetId, oldPage.getPageX(), oldPage.getPageY(), (int)oldPage.getZoomLevel());
                            BDBQuotaStore.this.pageById.put(transaction, (Object)newPage);
                            PageStats pageStats = (PageStats)BDBQuotaStore.this.pageStatsByPageId.get((Object)oldPageId);
                            if (pageStats == null) continue;
                            pageStats.setPageId(newPage.getId());
                            BDBQuotaStore.this.pageStatsById.putNoReturn(transaction, (Object)pageStats);
                        }
                    }
                }
            }
        }
    }

    private class Deleter
    implements Callable<Void> {
        private final String layerName;
        Predicate<TileSet> shouldDelete;

        public Deleter(String layerName, Predicate<TileSet> shouldDelete) {
            this.layerName = layerName;
            this.shouldDelete = shouldDelete;
        }

        @Override
        public Void call() throws Exception {
            Transaction transaction = BDBQuotaStore.this.entityStore.getEnvironment().beginTransaction(null, null);
            try {
                this.call(transaction);
                transaction.commit();
            }
            catch (RuntimeException e) {
                transaction.abort();
                throw e;
            }
            return null;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void call(Transaction transaction) {
            try (EntityCursor tileSets = BDBQuotaStore.this.tileSetsByLayer.entities(transaction, (Object)this.layerName, true, (Object)this.layerName, true, null);){
                TileSet tileSet;
                while (null != (tileSet = (TileSet)tileSets.next())) {
                    if (!this.shouldDelete.test(tileSet)) continue;
                    Quota freed = (Quota)BDBQuotaStore.this.usedQuotaByTileSetId.get(transaction, (Object)tileSet.getId(), LockMode.DEFAULT);
                    Quota global = (Quota)BDBQuotaStore.this.usedQuotaByTileSetId.get(transaction, (Object)BDBQuotaStore.GLOBAL_QUOTA_NAME, LockMode.DEFAULT);
                    tileSets.delete();
                    global.subtract(freed.getBytes());
                    BDBQuotaStore.this.usedQuotaById.put(transaction, (Object)global);
                }
            }
        }
    }

    private final class UsedQuotaByTileSetId
    implements Callable<Quota> {
        private final String tileSetId;

        private UsedQuotaByTileSetId(String tileSetId) {
            this.tileSetId = tileSetId;
        }

        @Override
        public Quota call() throws Exception {
            Quota quota = (Quota)BDBQuotaStore.this.usedQuotaByTileSetId.get(null, (Object)this.tileSetId, LockMode.READ_COMMITTED);
            if (quota == null) {
                quota = new Quota();
            }
            return quota;
        }
    }

    private class GetLayerNames
    implements Callable<Set<String>> {
        private GetLayerNames() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public Set<String> call() throws Exception {
            HashSet<String> names = new HashSet<String>();
            try (EntityCursor layerNameCursor = BDBQuotaStore.this.tileSetsByLayer.keys(null, CursorConfig.DEFAULT);){
                String name;
                while ((name = (String)layerNameCursor.nextNoDup()) != null) {
                    if (BDBQuotaStore.GLOBAL_QUOTA_NAME.equals(name)) continue;
                    names.add(name);
                }
            }
            return names;
        }
    }

    private class StartUpInitializer
    implements Callable<Void> {
        private StartUpInitializer() {
        }

        @Override
        public Void call() throws Exception {
            Transaction transaction = BDBQuotaStore.this.entityStore.getEnvironment().beginTransaction(null, null);
            try {
                if (null == BDBQuotaStore.this.usedQuotaByTileSetId.get(transaction, (Object)BDBQuotaStore.GLOBAL_QUOTA_NAME, LockMode.DEFAULT)) {
                    log.fine("First time run: creating global quota object");
                    TileSet globalTileSet = new TileSet(BDBQuotaStore.GLOBAL_QUOTA_NAME);
                    BDBQuotaStore.this.tileSetById.put(transaction, (Object)globalTileSet);
                    Quota globalQuota = new Quota();
                    globalQuota.setTileSetId(BDBQuotaStore.GLOBAL_QUOTA_NAME);
                    BDBQuotaStore.this.usedQuotaById.put(transaction, (Object)globalQuota);
                    log.fine("created Global Quota");
                }
                Set layerNames = BDBQuotaStore.this.tilePageCalculator.getLayerNames();
                Object existingLayers = new GetLayerNames().call();
                HashSet layersToDelete = new HashSet(existingLayers);
                layersToDelete.removeAll(layerNames);
                for (String layerName : layersToDelete) {
                    log.info("Deleting disk quota information for layer '" + layerName + "' as it does not exist anymore...");
                    try {
                        new Deleter(layerName, ts -> true).call(transaction);
                    }
                    catch (Exception e) {
                        log.log(Level.WARNING, "Error deleting disk quota information for layer '" + layerName + "'", e);
                    }
                }
                for (String layerName : layerNames) {
                    BDBQuotaStore.this.createLayer(layerName, transaction);
                }
                transaction.commit();
            }
            catch (RuntimeException e) {
                transaction.abort();
                throw e;
            }
            return null;
        }
    }
}

