/*
 * Decompiled with CFR 0.152.
 */
package org.geowebcache.sqlite;

import java.io.File;
import java.nio.file.Files;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.io.FileUtils;
import org.geotools.util.logging.Logging;
import org.geowebcache.sqlite.ExtendedConnection;
import org.geowebcache.sqlite.SqliteInfo;
import org.geowebcache.sqlite.Utils;

public final class SqliteConnectionManager {
    private static Logger LOGGER = Logging.getLogger((String)SqliteConnectionManager.class.getName());
    private final ConcurrentHashMap<File, PooledConnection> pool = new ConcurrentHashMap();
    private volatile boolean stopPoolReaper = false;

    public SqliteConnectionManager(SqliteInfo configuration) {
        this(configuration.getPoolSize(), configuration.getPoolReaperIntervalMs());
    }

    SqliteConnectionManager(long poolSize, long poolReaperIntervalMs) {
        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.info(String.format("Initiating connection poll: [poolSize='%d', poolReaperIntervalMs='%d'].", poolSize, poolReaperIntervalMs));
        }
        try {
            Class.forName("org.sqlite.JDBC");
        }
        catch (Exception exception) {
            throw Utils.exception(exception, "Error initiating sqlite driver.", new Object[0]);
        }
        double poolSizeThreshold = (double)poolSize * 0.9;
        double connectionsToRemove = (double)poolSize * 0.1;
        new Thread(() -> {
            while (!this.stopPoolReaper) {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine(String.format("Current pool size is '%d' and threshold is '%f'.", this.pool.size(), poolSizeThreshold));
                }
                if ((double)this.pool.size() > poolSizeThreshold) {
                    if (LOGGER.isLoggable(Level.INFO)) {
                        LOGGER.info(String.format("Reaping connections, current pool size %d.", this.pool.size()));
                    }
                    ArrayList<PooledConnection> pooledConnections = new ArrayList<PooledConnection>(this.pool.values());
                    Collections.sort(pooledConnections);
                    for (int i = 0; (double)i < connectionsToRemove && i < pooledConnections.size(); ++i) {
                        ((PooledConnection)pooledConnections.get(i)).reapConnection();
                    }
                }
                try {
                    Thread.sleep(poolReaperIntervalMs);
                }
                catch (Exception exception) {
                    Thread.currentThread().interrupt();
                    if (!LOGGER.isLoggable(Level.WARNING)) continue;
                    LOGGER.warning("Pool reaper interrupted.");
                }
            }
            if (LOGGER.isLoggable(Level.WARNING)) {
                LOGGER.warning("Pool reaper stop.");
            }
        }).start();
    }

    void executeSql(File file, String sql, Object ... parameters) {
        this.doWork(file, false, (Connection connection) -> this.executeSql(connection, sql, parameters));
    }

    void executeSql(Connection connection, String sql, Object ... parameters) {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine(String.format("Executing SQL '%s'.", sql));
        }
        try (PreparedStatement statement = connection.prepareStatement(sql);){
            for (int i = 0; i < parameters.length; ++i) {
                statement.setObject(i + 1, parameters[i]);
            }
            statement.execute();
        }
        catch (Exception exception) {
            throw Utils.exception(exception, "Error executing SQL '%s'.", sql);
        }
    }

    <T> T executeQuery(File file, ResultExtractor<T> extractor, String query, Object ... parameters) {
        return (T)this.doWork(file, true, (Connection connection) -> this.executeQuery(connection, extractor, query, parameters));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    <T> T executeQuery(Connection connection, ResultExtractor<T> extractor, String query, Object ... parameters) {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine(String.format("Executing query '%s'.", query));
        }
        try (PreparedStatement statement = connection.prepareStatement(query);){
            for (int i = 0; i < parameters.length; ++i) {
                statement.setObject(i + 1, parameters[i]);
            }
            T t = extractor.extract(statement.executeQuery());
            return t;
        }
        catch (Exception exception) {
            throw Utils.exception(exception, "Error executing query '%s'.", query);
        }
    }

    void doWork(File file, boolean readOnly, Work work) {
        this.doWork(file, readOnly, (Connection connection) -> {
            work.doWork(connection);
            return null;
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    <T> T doWork(File file, boolean readOnly, WorkWithResult<T> work) {
        if (readOnly) {
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine(String.format("Starting work on file '%s' in readonly mode.", file));
            }
        } else if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine(String.format("Starting work on file '%s' in write mode.", file));
        }
        PooledConnection pooledConnection = this.getPooledConnection(file);
        pooledConnection = readOnly ? pooledConnection.getReadLockOnValidConnection() : pooledConnection.getWriteLockOnValidConnection();
        ExtendedConnection connection = pooledConnection.getExtendedConnection();
        try {
            T result = work.doWork(connection);
            if (!connection.closeInvoked() && LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine("Close was not invoked on extended connection.");
            }
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine(String.format("Work on file '%s' is done.", file));
            }
            T t = result;
            return t;
        }
        finally {
            if (readOnly) {
                pooledConnection.releaseReadLock();
            } else {
                pooledConnection.releaseWriteLock();
            }
        }
    }

    void replace(File currentFile, File newFile) {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine(String.format("Replacing file '%s' with file '%s'.", currentFile, newFile));
        }
        PooledConnection currentPooledConnection = this.getPooledConnection(currentFile).getWriteLockOnValidConnection();
        try {
            currentPooledConnection.reapConnection();
            if (currentFile.exists()) {
                Files.delete(currentFile.toPath());
            }
            FileUtils.moveFile((File)newFile, (File)currentFile);
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.info(String.format("File '%s' replaced with file '%s'.", currentFile, newFile));
            }
        }
        catch (Exception exception) {
            throw Utils.exception(exception, "Error replacing file '%s' with file '%s'.", currentFile, newFile);
        }
        finally {
            currentPooledConnection.releaseWriteLock();
        }
    }

    void delete(File file) {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine(String.format("Deleting file '%s'.", file));
        }
        if (!file.exists()) {
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine(String.format("File '%s' doesn't exists.", file));
            }
            return;
        }
        PooledConnection pooledConnection = this.getPooledConnection(file).getWriteLockOnValidConnection();
        try {
            pooledConnection.closeConnection();
            FileUtils.deleteQuietly((File)file);
            this.pool.remove(file);
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.info(String.format("File '%s' deleted.", file));
            }
        }
        catch (Exception exception) {
            throw Utils.exception(exception, "Error deleting file '%s'.", file);
        }
        finally {
            pooledConnection.releaseWriteLock();
        }
    }

    void rename(File currentFile, File newFile) {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine(String.format("Renaming file '%s' to '%s'.", currentFile, newFile));
        }
        PooledConnection pooledConnection = this.getPooledConnection(currentFile).getWriteLockOnValidConnection();
        try {
            pooledConnection.closeConnection();
            this.pool.remove(currentFile);
            FileUtils.moveFile((File)currentFile, (File)newFile);
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.info(String.format("File '%s' renamed to '%s'.", currentFile, newFile));
            }
        }
        catch (Exception exception) {
            throw Utils.exception(exception, "Renaming file '%s' to '%s'.", currentFile, newFile);
        }
        finally {
            pooledConnection.releaseWriteLock();
        }
    }

    public Map<File, PooledConnection> getPool() {
        return this.pool;
    }

    void reapAllConnections() {
        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.info("Reaping all connections.");
        }
        this.pool.values().forEach(PooledConnection::reapConnection);
    }

    void stopPoolReaper() {
        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.info("Stopping the pool reaper.");
        }
        this.stopPoolReaper = true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private PooledConnection getPooledConnection(File file) {
        try {
            PooledConnection pooledConnection = this.pool.get(file);
            if (pooledConnection != null) {
                return pooledConnection;
            }
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine(String.format("Creating pooled connection to file '%s'.", file));
            }
            pooledConnection = new PooledConnection(file);
            pooledConnection.getWriteLock();
            try {
                PooledConnection existing = this.pool.putIfAbsent(file, pooledConnection);
                if (existing != null) {
                    if (LOGGER.isLoggable(Level.FINE)) {
                        LOGGER.fine(String.format("Connection to file '%s' already exists, closing this one.", file));
                    }
                    pooledConnection.closeConnection();
                    PooledConnection pooledConnection2 = existing;
                    return pooledConnection2;
                }
                pooledConnection.init();
                PooledConnection pooledConnection3 = pooledConnection;
                return pooledConnection3;
            }
            finally {
                pooledConnection.releaseWriteLock();
            }
        }
        catch (Exception exception) {
            throw Utils.exception(exception, "Error opening connection to file '%s'.", file);
        }
    }

    private final class PooledConnection
    implements Comparable<PooledConnection> {
        private final File file;
        private Connection connection;
        private final ReentrantReadWriteLock lock;
        private long lastAccess;
        private volatile boolean closed;

        PooledConnection(File file) {
            this.file = file;
            this.lock = new ReentrantReadWriteLock();
            this.closed = true;
        }

        void init() {
            this.connection = this.openConnection(this.file);
            this.lastAccess = System.currentTimeMillis();
            this.closed = false;
        }

        @Override
        public int compareTo(PooledConnection other) {
            if (this.lastAccess < other.lastAccess) {
                return -1;
            }
            return 1;
        }

        ExtendedConnection getExtendedConnection() {
            this.lastAccess = System.currentTimeMillis();
            return new ExtendedConnection(this.connection);
        }

        void reapConnection() {
            this.getWriteLock();
            this.closeConnection();
            SqliteConnectionManager.this.pool.remove(this.file);
            this.releaseWriteLock();
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.info(String.format("Connection to file '%s' reaped.", this.file));
            }
        }

        void closeConnection() {
            if (!this.closed) {
                try {
                    this.connection.close();
                    this.closed = true;
                }
                catch (Exception exception) {
                    throw Utils.exception("Error closing connection to file '%s'.", this.file);
                }
                if (LOGGER.isLoggable(Level.INFO)) {
                    LOGGER.info(String.format("Connection to file '%s' closed.", this.file));
                }
            }
        }

        void getReadLock() {
            String logId = "";
            if (LOGGER.isLoggable(Level.FINE)) {
                logId = UUID.randomUUID().toString();
                LOGGER.fine(String.format("[%s] Waiting for read lock on file '%s'.", logId, this.file));
            }
            this.lock.readLock().lock();
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine(String.format("[%s] Read lock on file '%s' obtained.", logId, this.file));
            }
        }

        PooledConnection getReadLockOnValidConnection() {
            this.getReadLock();
            if (!this.closed) {
                return this;
            }
            this.releaseReadLock();
            for (int i = 0; i < 10; ++i) {
                PooledConnection pooledConnection = SqliteConnectionManager.this.getPooledConnection(this.file);
                pooledConnection.getReadLock();
                if (pooledConnection.closed) continue;
                return pooledConnection;
            }
            throw Utils.exception("Could not obtain a valid connection to file '%s'.", this.file);
        }

        void releaseReadLock() {
            this.lock.readLock().unlock();
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine(String.format("Read lock on file '%s' released.", this.file));
            }
        }

        void getWriteLock() {
            String logId = "";
            if (LOGGER.isLoggable(Level.FINE)) {
                logId = UUID.randomUUID().toString();
                LOGGER.fine(String.format("[%s] Waiting for write lock on file '%s'.", logId, this.file));
            }
            this.lock.writeLock().lock();
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine(String.format("[%s] Write lock on file '%s' obtained.", logId, this.file));
            }
        }

        PooledConnection getWriteLockOnValidConnection() {
            this.getWriteLock();
            if (!this.closed) {
                return this;
            }
            this.releaseWriteLock();
            for (int i = 0; i < 10; ++i) {
                PooledConnection pooledConnection = SqliteConnectionManager.this.getPooledConnection(this.file);
                pooledConnection.getWriteLock();
                if (pooledConnection.closed) continue;
                return pooledConnection;
            }
            throw Utils.exception("Could not obtain a valid connection to file '%s'.", this.file);
        }

        void releaseWriteLock() {
            this.lock.writeLock().unlock();
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine(String.format("Write lock on file '%s' released.", this.file));
            }
        }

        private Connection openConnection(File file) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.info(String.format("Opening connection to file '%s'.", file));
            }
            Utils.createFileParents(file);
            try {
                return DriverManager.getConnection("jdbc:sqlite:" + file.getPath());
            }
            catch (Exception exception) {
                throw Utils.exception(exception, "Error opening connection to file '%s'.", file);
            }
        }
    }

    static interface ResultExtractor<T> {
        public T extract(ResultSet var1) throws SQLException;
    }

    static interface WorkWithResult<T> {
        public T doWork(Connection var1);
    }

    static interface Work {
        public void doWork(Connection var1);
    }
}

